#!/bin/bash

set -a

PERSISTENCE_ENABLED=${PERSISTENCE_ENABLED:-"false"}
DATA_DIR=${DATA_DIR:-"/data"}
EXTERNAL_CONFIG_FILE=${EXTERNAL_CONFIG_FILE:-"/etc/rcache/external.conf.d/rcache-additional.conf"}
RCACHE_MAJOR_VERSION=${RCACHE_MAJOR_VERSION:-"v7"}

apply_permissions() {
    chgrp -R 1000 /etc/rcache
    chmod -R g=u /etc/rcache
}

common_operation() {
    mkdir -p "${DATA_DIR}"
    mkdir -p "/etc/rcache/external.conf.d"
    touch -p "/etc/rcache/external.conf.d/rcache-additional.conf"
}

set_rcache_password() {
    if [[ -z "${RCACHE_PASSWORD}" ]]; then
        echo "rcache is running without password which is not recommended"
        echo "protected-mode no" >> /etc/rcache/rcache.conf
    else
        {
            echo masterauth "${RCACHE_PASSWORD}"
            echo requirepass "${RCACHE_PASSWORD}"
            echo protected-mode yes
        } >> /etc/rcache/rcache.conf
    fi
}

rcache_mode_setup() {
    if [[ "${SETUP_MODE}" == "cluster" ]]; then
        {
            echo cluster-enabled yes
            echo cluster-require-full-coverage no
            echo cluster-migration-barrier 1
            echo cluster-config-file "${DATA_DIR}/nodes.conf"
        } >> /etc/rcache/rcache.conf

        POD_IP=$(hostname -i)
        sed -i -e "/myself/ s/[0-9]\{1,3\}\.[0-9]\{1,3\}\.[0-9]\{1,3\}\.[0-9]\{1,3\}/${POD_IP}/" "${DATA_DIR}/nodes.conf"
    else
        echo "Setting up rcache in standalone mode"
    fi
}

tls_setup() {
    if [[ "${TLS_MODE}" == "true" ]]; then
        {
            echo port 0
            echo tls-port 6379
            echo tls-cert-file "${RCACHE_TLS_CERT}"
            echo tls-key-file "${RCACHE_TLS_CERT_KEY}"
            echo tls-ca-cert-file "${RCACHE_TLS_CA_KEY}"
            # echo tls-prefer-server-ciphers yes
            echo tls-auth-clients optional
        } >> /etc/rcache/rcache.conf

        if [[ "${SETUP_MODE}" == "cluster" ]]; then
            {
                echo tls-replication yes
                echo tls-cluster yes
            } >> /etc/rcache/rcache.conf
        fi
    else
        echo "Running without TLS mode"
    fi
}

persistence_setup() {
    if [[ "${PERSISTENCE_ENABLED}" == "true" ]]; then
        {
            echo save 900 1
            echo save 300 10
            echo save 60 10000
            echo appendonly yes
            echo appendfilename \"appendonly.aof\"
            echo dir "${DATA_DIR}"
        } >> /etc/rcache/rcache.conf
    else
        echo "Running without persistence mode"
    fi
}

external_config() {
    echo "include ${EXTERNAL_CONFIG_FILE}" >> /etc/rcache/rcache.conf
}

start_rcache() {
    if [[ "${SETUP_MODE}" == "cluster" ]]; then
        echo "Starting rcache service in cluster mode....."
        if [[ "${RCACHE_MAJOR_VERSION}" != "v7" ]]; then
          rcache-server /etc/rcache/rcache.conf --cluster-announce-ip "${POD_IP}"
        else
          rcache-server /etc/rcache/rcache.conf
        fi
    else
        echo "Starting rcache service in standalone mode....."
        rcache-server /etc/rcache/rcache.conf
    fi
}

main_function() {
    common_operation
    set_rcache_password
    rcache_mode_setup
    persistence_setup
    tls_setup
    external_config
    start_rcache
}

main_function
