package controller

import (
	"errors"
	"github.com/360EntSecGroup-Skylar/excelize"
	"github.com/gin-gonic/gin"
	json "github.com/json-iterator/go"
	"github.com/spf13/cast"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/bean/vo/request"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/pkg/beagle/resp"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/service"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/util"
)

// ParsingHostFiles 解析主机文件
func ParsingHostFiles(c *gin.Context) (hostManageList []request.HostManageList, err error) {
	//获取文件流
	metaData, _, err := c.Request.FormFile("host_file")
	if err != nil {
		err = resp.GetFileStreamError.WithError(err)
		return
	}
	xlsxData, err := excelize.OpenReader(metaData)
	if err != nil {
		err = resp.ReadFileError.WithError(err)
		return
	}
	//读取工作簿
	rows := xlsxData.GetRows("Sheet1")

	if len(rows) > 1001 {
		err = resp.FailedToParseFile.WithError(errors.New("ip数量不可超过1000"))
		return
	}

	for i := 0; i < len(rows); i++ {
		//默认跳过第一行
		if i < 1 {
			continue
		}

		var voucherType int
		if rows[i][2] == "密码验证" {
			voucherType = 0
		} else {
			voucherType = 1
		}
		hostManageList = append(hostManageList, request.HostManageList{
			Ip:          rows[i][0],
			Port:        rows[i][1],
			VoucherType: voucherType,
			UserName:    rows[i][3],
			Password:    rows[i][4],
		})
	}
	return
}

// GetMinioFiles 解析minio中xlsx类型文件
func GetMinioFiles(fileName string) (hostManageList []request.HostManageList, err error) {

	obj, err := service.DocLibSvc{}.GetFile(fileName)
	if err != nil {
		err = resp.GetFileStreamError.WithError(err)
		return
	}

	xlsxData, err := excelize.OpenReader(obj)
	if err != nil {
		err = resp.ReadFileError.WithError(err)
		return
	}
	//读取工作簿
	rows := xlsxData.GetRows("Sheet1")

	if len(rows) > 1003 {
		err = resp.FailedToParseFile.WithError(errors.New("ip数量不可超过1000"))
		return
	}

	for i := 0; i < len(rows); i++ {
		//默认跳过前两行
		if i < 2 {
			continue
		}

		var voucherType int
		if rows[i][2] == "密码验证" {
			voucherType = 0
		} else {
			voucherType = 1
		}
		hostManageList = append(hostManageList, request.HostManageList{
			Ip:          rows[i][0],
			Port:        rows[i][1],
			VoucherType: voucherType,
			UserName:    rows[i][3],
			Password:    rows[i][4],
		})
	}
	return

}

// AddHostManage 新增
func AddHostManage(c *gin.Context) {
	var (
		req request.AddHostManageReq
		err error
	)
	if err = c.ShouldBindJSON(&req); err != nil {
		SendJsonResponse(c, resp.InvalidParam.WithError(err), nil)
		return
	}

	//参数校验
	if req.HostName == "" {
		SendJsonResponse(c, resp.InvalidParam.WithError(errors.New("主机分组名称为空")), nil)
		return
	}
	//if len(req.HostManageList) == 0 {
	//	SendJsonResponse(c, resp.InvalidParam.WithError(errors.New("主机分组数量为0")), nil)
	//	return
	//}
	//for _, v := range req.HostManageList {
	//	if v.Ip == "" || v.UserName == "" {
	//		SendJsonResponse(c, resp.InvalidParam.WithError(errors.New("ip或用户名为空")), nil)
	//		return
	//	}
	//	if v.VoucherType == 0 && v.Password == "" {
	//		SendJsonResponse(c, resp.InvalidParam.WithError(errors.New("密码为空")), nil)
	//		return
	//	}
	//}

	hostManageSvc := service.HostManageSvc{}
	err = hostManageSvc.AddHostManage(req)
	if err != nil {
		SendJsonResponse(c, err, nil)
		return
	}
	SendJsonResponse(c, resp.OK, nil)
}

// EditHostManage 编辑
func EditHostManage(c *gin.Context) {
	var (
		req request.EditHostManageReq
		err error
	)
	if err = c.ShouldBind(&req); err != nil {
		SendJsonResponse(c, resp.InvalidParam.WithError(err), nil)
		return
	}

	//参数校验
	//if len(req.HostManageList) == 0 {
	//	SendJsonResponse(c, resp.InvalidParam.WithError(errors.New("主机分组数量为0")), nil)
	//	return
	//}
	//for _, v := range req.HostManageList {
	//	if v.Ip == "" || v.UserName == "" {
	//		SendJsonResponse(c, resp.InvalidParam.WithError(errors.New("ip或用户名为空")), nil)
	//		return
	//	}
	//	if v.VoucherType == 0 && v.Password == "" {
	//		SendJsonResponse(c, resp.InvalidParam.WithError(errors.New("密码为空")), nil)
	//		return
	//	}
	//}

	hostManageSvc := service.HostManageSvc{}
	err = hostManageSvc.EditHostManage(req)
	if err != nil {
		SendJsonResponse(c, err, nil)
		return
	}
	SendJsonResponse(c, resp.OK, nil)
}

// DelHostManage 删除
func DelHostManage(c *gin.Context) {
	var req request.DelHostManageReq
	if err := c.ShouldBindJSON(&req); err != nil {
		SendJsonResponse(c, resp.InvalidParam.WithError(err), nil)
		return
	}
	//参数校验
	if err := util.ValidateSimple(req, "Id"); err != nil {
		SendJsonResponse(c, resp.InvalidParam.WithError(err), nil)
		return
	}

	hostManageSvc := service.HostManageSvc{}
	err := hostManageSvc.DelHostManage(req)
	if err != nil {
		SendJsonResponse(c, err, nil)
		return
	}
	SendJsonResponse(c, resp.OK, nil)
}

// DetailsHostManage 详情
func DetailsHostManage(c *gin.Context) {
	var (
		err error
		id  string
	)

	if id = c.Query("id"); id == "" {
		id = c.Param("id")
	}
	if id == "" {
		SendJsonResponse(c, resp.InvalidParam.WithError(errors.New("id为空")), nil)
		return
	}

	hostManageSvc := service.HostManageSvc{}
	data, err := hostManageSvc.DetailsHostManage(cast.ToInt(id))
	if err != nil {
		SendJsonResponse(c, err, nil)
		return
	}
	SendJsonResponse(c, resp.OK, data)
}

// PageListHostManage 列表-分页
func PageListHostManage(c *gin.Context) {
	var req request.ListHostManageReq
	if err := c.ShouldBind(&req); err != nil {
		SendJsonResponse(c, resp.InvalidParam.WithError(err), nil)
		return
	}

	hostManageSvc := service.HostManageSvc{}
	total, list, err := hostManageSvc.PageListHostManage(req)
	if err != nil {
		SendJsonPageResponse(c, err, nil, 0)
		return
	}
	SendJsonPageResponse(c, resp.OK, list, total)
}

// StateHostManage 状态检测
func StateHostManage(c *gin.Context) {
	var (
		req request.StateHostManageReq
		err error
	)
	if err = c.ShouldBind(&req); err != nil {
		SendJsonResponse(c, resp.InvalidParam.WithError(err), nil)
		return
	}

	hostManageList := make([]request.HostManageList, 0)
	hostManageSvc := service.HostManageSvc{}

	switch req.DetectionType {
	case 1:
		if req.Id == 0 {
			SendJsonResponse(c, resp.InvalidParam.WithError(errors.New("id为空")), nil)
			return
		}
		//列表点击状态检测
		err = hostManageSvc.ListStateHostManage(req)
		if err != nil {
			SendJsonResponse(c, err, nil)
			return
		}
		SendJsonResponse(c, resp.OK, nil)
		return
	case 2:
		//解析主机列表
		if req.HostManageList == "" {
			SendJsonResponse(c, resp.InvalidParam.WithError(errors.New("主机IP列表为空")), nil)
			return
		}
		err = json.Unmarshal([]byte(req.HostManageList), &hostManageList)
		if err != nil {
			SendJsonResponse(c, resp.InvalidParam.WithError(err), nil)
			return
		}
	case 3:
		//解析主机文件
		//hostManageList, err = ParsingHostFiles(c)
		hostManageList, err = GetMinioFiles(req.FileName)

		if err != nil {
			SendJsonResponse(c, resp.ReadFileError.WithError(errors.New("解析主机文件失败")), nil)
			return
		}
	}

	//参数校验
	if len(hostManageList) == 0 {
		SendJsonResponse(c, resp.InvalidParam.WithError(errors.New("主机分组数量为0")), nil)
		return
	}
	if len(hostManageList) > 5 {
		SendJsonResponse(c, resp.InvalidParam.WithError(errors.New("主机分组ip数量超过5条")), nil)
		return
	}
	for _, v := range hostManageList {
		if v.Ip == "" || v.UserName == "" {
			SendJsonResponse(c, resp.InvalidParam.WithError(errors.New("ip或用户名为空")), nil)
			return
		}
		if v.VoucherType == 0 && v.Password == "" {
			SendJsonResponse(c, resp.InvalidParam.WithError(errors.New("密码为空")), nil)
			return
		}
	}
	//保存时状态检测
	id, err := hostManageSvc.SaveStateHostManage(hostManageList)
	if err != nil {
		SendJsonResponse(c, err, id)
		return
	}
	SendJsonResponse(c, resp.OK, id)

}

// HostIpExceptionList 主机ip异常列表
func HostIpExceptionList(c *gin.Context) {
	var req request.HostIpExceptionListReq
	if err := c.ShouldBind(&req); err != nil {
		SendJsonResponse(c, resp.InvalidParam.WithError(err), nil)
		return
	}

	hostManageSvc := service.HostManageSvc{}
	switch req.DetectionType {
	case 1:
		//列表页异常列表
		if req.Id == 0 {
			SendJsonResponse(c, resp.InvalidParam.WithError(errors.New("id为空")), nil)
			return
		}
		//列表点击状态检测
		total, list, err := hostManageSvc.HostIpExceptionList(req)
		if err != nil {
			SendJsonPageResponse(c, err, nil, 0)
			return
		}
		SendJsonPageResponse(c, err, list, total)

	case 2:
		//保存时异常列表
		if req.Uuid == "" {
			SendJsonResponse(c, resp.InvalidParam.WithError(errors.New("uuid为空")), nil)
			return
		}
		//保存时状态检测异常列表
		total, list, err := hostManageSvc.SaveIpExceptionList(req)
		if err != nil {
			SendJsonPageResponse(c, err, nil, 0)
			return
		}
		SendJsonPageResponse(c, err, list, total)
	}
}

// 导出
func ExportIp(c *gin.Context) {
	detectionType := c.Query("detection_type")
	id := c.Query("id")
	uuid := c.Query("uuid")

	if cast.ToInt(detectionType) != 0 && cast.ToInt(detectionType) != 1 {
		SendJsonResponse(c, resp.InvalidParam.WithError(errors.New("导出类型解析错误")), nil)
		return
	}
	if cast.ToInt(detectionType) == 1 && id == "" {
		SendJsonResponse(c, resp.InvalidParam.WithError(errors.New("id为空")), nil)
		return
	}
	if cast.ToInt(detectionType) == 0 && uuid == "" {
		SendJsonResponse(c, resp.InvalidParam.WithError(errors.New("uuid为空")), nil)
		return
	}

	hostManageSvc := service.HostManageSvc{}
	fileName, err := hostManageSvc.ExportIp(id, uuid, cast.ToInt(detectionType))
	if err != nil {
		SendJsonResponse(c, err, nil)
		return
	}
	c.Writer.Header().Add("Content-Type", "application/octet-stream")
	c.Writer.Header().Add("Content-Transfer-Encoding", "binary")
	c.Writer.Header().Add("Content-Disposition", "filename="+fileName)

	c.File("/app/xlsx/" + fileName)
}

// ListHostManage 列表
func ListHostManage(c *gin.Context) {

	hostManageSvc := service.HostManageSvc{}
	list, err := hostManageSvc.ListHostManage()
	if err != nil {
		SendJsonResponse(c, err, nil)
		return
	}
	SendJsonResponse(c, resp.OK, list)
}
