package controller

import (
	vd "github.com/bytedance/go-tagexpr/validator"
	"github.com/gin-gonic/gin"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/bean/vo/request"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/common/conf"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/pkg/beagle/resp"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/service"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/util"

	"go.uber.org/zap"
	"strconv"
)

func UserLogin(c *gin.Context) {
	userReq := request.UserReq{}
	err := c.BindJSON(&userReq)
	if err != nil {
		SendJsonResponse(c, resp.InvalidParam.WithError(err), nil)
		return
	}
	//参数检测
	if err := vd.Validate(userReq); err != nil {
		SendJsonResponse(c, resp.InvalidParam.WithError(err), "")
		return
	}
	// 验证 userName 是否含有危险字符
	if util.IfDangerCharacter(userReq.SystemAccount) {
		SendJsonResponse(c, resp.InvalidParam.WithMsg("账号存在危险字符"), "")
		return
	}
	//TODO 解密password
	pwd, err := util.DecryptPwd(userReq.Password)
	if err != nil {
		SendJsonResponse(c, resp.FAIL.WithError(err), "")
		return
	}
	userReq.Password = pwd
	loginInf := service.UserSvc{SystemAccount: userReq.SystemAccount, PassWord: userReq.Password}
	msg, uuidStr, lastLogin, err := loginInf.Login()
	if err != nil {
		SendJsonResponse(c, err, "")
		return
	}
	c.SetCookie(conf.CookieName, uuidStr, 1*60*60*24, "/", "", false, false)
	c.SetCookie(conf.CookieNameLastLogin, lastLogin, 1*60*60*24, "/", "", false, false)
	SendJsonResponse(c, resp.OK, msg)
}

func GetUserInfo(c *gin.Context) {
	token, _ := c.Cookie(conf.CookieName)
	conf.Logger.Info("当前token信息为", zap.String("bgToken", token))
	header := c.GetHeader(conf.CookieName)
	conf.Logger.Info("当前header信息为", zap.String("header", header))
	if token == "" {
		SendJsonResponse(c, resp.FAIL.WithMsg("登录超时"), "")
		return
	}
	loginInf := service.UserSvc{}
	result, err := loginInf.GetCurUser(token)
	if err != nil {
		SendJsonResponse(c, err, "")
		return
	}
	SendJsonResponse(c, resp.OK, result)
}

func GetCaptcha(c *gin.Context) {
	widthS := c.Query("width")
	heightS := c.Query("height")
	var width, height int
	if widthS == "" {
		width = conf.WIDTH
	} else {
		width, _ = strconv.Atoi(widthS)
	}
	if heightS == "" {
		height = conf.HEIGHT
	} else {
		height, _ = strconv.Atoi(heightS)
	}
	loginInf := service.UserSvc{}
	m, err := loginInf.GetCaptcha(width, height)
	if err != nil {
		SendJsonResponse(c, err, "")
		return
	}
	SendJsonResponse(c, resp.OK, m)
}

func VerifyCaptcha(c *gin.Context) {
	id := c.Query("id")
	value := c.Query("value")
	if id == "" || value == "" {
		SendJsonResponse(c, resp.FAIL.WithMsg("id and value are required"), "")
		return
	}
	loginInf := service.UserSvc{}
	err := loginInf.VerifyCaptcha(id, value)
	if err != nil {
		SendJsonResponse(c, err, "")
		return
	}
	SendJsonResponse(c, resp.OK, "验证码正确")
}

func UserLogout(c *gin.Context) {
	cookie, _ := c.Cookie(conf.CookieName)
	c.SetCookie(conf.CookieName, "0", -1, "/", "", false, true)
	c.SetCookie(conf.CookieNameLastLogin, "0", -1, "/", "", false, true)
	loginInf := service.UserSvc{}
	msg, err := loginInf.UserLogout(cookie)
	if err != nil {
		SendJsonResponse(c, err, "")
		return
	}
	SendJsonResponse(c, resp.OK, msg)
}
