package controller

import (
	"errors"
	vd "github.com/bytedance/go-tagexpr/validator"
	"github.com/gin-gonic/gin"
	"github.com/spf13/cast"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/bean/vo/request"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/pkg/beagle/resp"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/router/middleware/header"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/service"
)

// 查询组织用户详情
func OrgUserDetail(c *gin.Context) {
	id := cast.ToInt(c.Param("id"))
	if id <= 0 {
		SendJsonResponse(c, resp.InvalidParam.WithMsg("请输入用户id"), nil)
		return
	}
	svc := service.User{}
	result, err := svc.OrgUserDetail(id)
	if err != nil {
		SendJsonResponse(c, err, nil)
		return
	}
	SendJsonResponse(c, resp.OK, result)
}

// 组织添加用户时查询的角色列表
func OrgUserRoles(c *gin.Context) {
	oid := c.Query("oid")
	if oid == "" {
		SendJsonResponse(c, resp.InvalidParam.WithMsg("组织id必填"), nil)
		return
	}
	svc := service.User{}
	result, err := svc.OrgUserRoles(oid)
	if err != nil {
		SendJsonResponse(c, err, nil)
		return
	}
	SendJsonResponse(c, resp.OK, result)
}

// 添加组织用户
func OrgAddUser(c *gin.Context) {
	var input request.OrgUserInput
	if err := c.ShouldBindJSON(&input); err != nil {
		SendJsonResponse(c, resp.InvalidParam.WithError(err), nil)
		return
	}
	if err := vd.Validate(input); err != nil {
		SendJsonResponse(c, err, "")
		return
	}

	svc := service.User{User: header.GetUser(c)}
	if err := svc.OrgAddUser(input); err != nil {
		SendJsonResponse(c, err, nil)
		return
	}
	SendJsonResponse(c, resp.OK, nil)
}

// 编辑用户
func OrgUpdateUser(c *gin.Context) {
	id := cast.ToInt(c.Param("id"))
	if id <= 0 {
		SendJsonResponse(c, resp.InvalidParam.WithMsg("请输入用户id"), nil)
		return
	}
	var input request.UpdateOrgUserInput
	if err := c.ShouldBindJSON(&input); err != nil {
		SendJsonResponse(c, resp.FAIL.WithError(err), nil)
		return
	}
	if err := vd.Validate(input); err != nil {
		SendJsonResponse(c, err, "")
		return
	}
	svc := service.User{User: header.GetUser(c)}
	if err := svc.OrgUpdateUser(id, input); err != nil {
		SendJsonResponse(c, err, nil)
		return
	}
	SendJsonResponse(c, resp.OK, nil)
}

// 删除组织用户
func DelOrgUser(c *gin.Context) {
	var input request.DelOrgUser
	if err := c.ShouldBindJSON(&input); err != nil {
		SendJsonResponse(c, resp.InvalidParam.WithError(err), nil)
		return
	}
	if len(input.Ids) <= 0 {
		SendJsonResponse(c, resp.FAIL.WithMsg("请输入用户id"), nil)
		return
	}
	svc := service.User{User: header.GetUser(c)}
	if err := svc.DelOrgUser(input); err != nil {
		SendJsonResponse(c, err, nil)
		return
	}
	SendJsonResponse(c, resp.OK, nil)
}

// 去重校验
func CheckRepetition(c *gin.Context) {
	var input request.CheckRepetition
	if err := c.ShouldBindJSON(&input); err != nil {
		SendJsonResponse(c, resp.InvalidParam.WithError(err), nil)
		return
	}
	svc := service.User{}
	if err := svc.CheckRepetition(input); err != nil {
		SendJsonResponse(c, err, nil)
		return
	}
	SendJsonResponse(c, resp.OK, nil)
}

// 运维人员获取
func DevOps(c *gin.Context) {
	svc := service.User{User: header.GetUser(c)}
	list, err := svc.DevOps()
	if err != nil {
		SendJsonResponse(c, err, nil)
		return
	}
	SendJsonResponse(c, resp.OK, list)
}

// SystemUserEditPassword 修改密码
func SystemUserEditPassword(c *gin.Context) {
	params := request.SystemUserEditPasswordReq{}
	if err := c.ShouldBindJSON(&params); err != nil {
		SendJsonResponse(c, resp.InvalidParam.WithError(err), nil)
		return
	}
	pwdLen := len([]rune(params.Password))
	if pwdLen <= 0 {
		SendJsonResponse(c, resp.InvalidParam.WithError(errors.New("请输入密码")), "")
	}
	if err := vd.Validate(params); err != nil {
		SendJsonResponse(c, resp.InvalidParam.WithError(err), "")
		return
	}
	err := service.SystemUserEditPassword(params)
	if err != nil {
		SendJsonResponse(c, err, nil)
		return
	}
	SendJsonResponse(c, resp.OK, nil)
}

// ResetSystemUserPassword 重置系统账户密码
func ResetSystemUserPassword(c *gin.Context) {
	params := request.ResetSystemUserPasswordReq{}
	if err := c.ShouldBindJSON(&params); err != nil {
		SendJsonResponse(c, resp.InvalidParam.WithError(err), nil)
		return
	}
	if err := vd.Validate(params); err != nil {
		SendJsonResponse(c, resp.InvalidParam.WithError(err), "")
		return
	}
	err := service.ResetSystemUserPassword(params)
	if err != nil {
		SendJsonResponse(c, err, nil)
		return
	}
	SendJsonResponse(c, resp.OK, nil)
}

func SystemUserUpdateState(c *gin.Context) {
	params := request.SystemUserStateReq{}
	if err := c.ShouldBindJSON(&params); err != nil {
		SendJsonResponse(c, resp.InvalidParam.WithError(err), nil)
		return
	}

	svc := service.User{User: header.GetUser(c)}
	err := svc.SystemUserUpdateState(params)
	if err != nil {
		SendJsonResponse(c, err, nil)
		return
	}
	SendJsonResponse(c, resp.OK, nil)
}
