package service

import (
	"errors"
	"fmt"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/bean/entity"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/bean/vo/request"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/bean/vo/response"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/common/client"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/common/conf"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/pkg/beagle/jsontime"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/pkg/beagle/resp"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/util"
	"time"
	"xorm.io/xorm"

	"go.uber.org/zap"
)

type SystemAccessRule struct {
	User entity.SystemUserInfo
}

// AddAccessRule 新增访问规则
func AddAccessRule(params request.AddSystemAccessRuleReq) (err error) {
	// 拼装 访问规则 字段信息
	systemAccessRule := entity.SystemAccessRule{
		RuleId:      util.GetUUID(),
		RuleName:    params.RuleName,
		State:       params.State,
		RuleType:    params.RuleType,
		RuleDetail:  params.RuleDetail,
		CreatedBy:   params.CreatedBy,
		UpdatedBy:   params.CreatedBy,
		CreatedTime: jsontime.Time(time.Now()),
		UpdatedTime: jsontime.Time(time.Now()),
	}

	// 拼装 规则适用 用户范围 字段信息
	systemRuleUsers := make([]entity.SystemRuleUser, 0)
	for _, v := range params.UserScope {
		systemRuleUser := entity.SystemRuleUser{
			UserId:      v.UserId,
			RuleId:      systemAccessRule.RuleId,
			CreatedTime: time.Now(),
		}
		systemRuleUsers = append(systemRuleUsers, systemRuleUser)
	}

	db, err := client.GetDbClient()
	if err != nil {
		err = resp.DbConnectError.ErrorDetail(err)
		return
	}

	// 校验 访问规则名称 是否重复
	has, err := db.Table("system_access_rule").Where("rule_name =?", params.RuleName).Exist()
	if has {
		conf.Logger.Error("规则名称重复", zap.Error(err))
		err = resp.DbSelectError.ErrorDetail(errors.New("规则名称重复"))
		return
	}
	_, err = db.Transaction(func(session *xorm.Session) (data interface{}, err error) {
		// 添加规则记录
		_, err = session.Insert(&systemAccessRule)
		if err != nil {
			conf.Logger.Error("创建访问规则失败", zap.Error(err))
			err = resp.DbInsertError.ErrorDetail(errors.New("创建访问规则失败"))
			return
		}

		_, err = session.Insert(&systemRuleUsers)
		if err != nil {
			conf.Logger.Error("创建访问规则用户范围失败", zap.Error(err))
			err = resp.DbInsertError.ErrorDetail(errors.New("创建访问规则用户范围失败"))
			return
		}
		return
	})
	if err != nil {
		return
	}
	return
}

// UpdateAccessRule 修改访问规则
func UpdateAccessRule(params request.UpdateAccessRuleReq) (err error) {
	// 拼装 访问规则 字段信息
	systemAccessRule := entity.SystemAccessRule{
		RuleName:    params.RuleName,
		State:       params.State,
		RuleType:    params.RuleType,
		RuleDetail:  params.RuleDetail,
		UpdatedBy:   params.UpdatedBy,
		UpdatedTime: jsontime.Time(time.Now()),
	}

	// 拼装 规则适用 用户范围 字段信息
	systemRuleUsers := make([]entity.SystemRuleUser, 0)
	for _, v := range params.UserScope {
		systemRuleUser := entity.SystemRuleUser{
			UserId:      v.UserId,
			RuleId:      params.RuleId,
			CreatedTime: time.Now(),
		}
		systemRuleUsers = append(systemRuleUsers, systemRuleUser)
	}

	db, err := client.GetDbClient()
	if err != nil {
		err = resp.DbConnectError.ErrorDetail(err)
		return
	}

	// 校验 访问规则名称 是否重复
	cnt, err := db.Table("system_access_rule").Where("rule_name =?", params.RuleName).Count()
	if cnt > 1 || err != nil {
		conf.Logger.Error("规则名称重复", zap.Error(err))
		err = resp.DbSelectError.ErrorDetail(errors.New("规则名称重复"))
		return
	}

	_, err = db.Transaction(func(session *xorm.Session) (data interface{}, err error) {
		//修改规则记录
		_, err = session.Where("rule_id = ?", params.RuleId).Cols("rule_name,state,rule_type,rule_detail,updated_by,updated_time").Update(&systemAccessRule)
		if err != nil {
			conf.Logger.Error("修改访问规则失败", zap.Error(err))
			err = resp.DbInsertError.ErrorDetail(errors.New("修改访问规则失败"))
			return
		}

		//删除原规则用户范围
		_, err = session.Where("rule_id = ?", params.RuleId).Delete(new(entity.SystemRuleUser))
		if err != nil {
			conf.Logger.Error("修改访问规则用户范围失败", zap.Error(err))
			err = resp.DbInsertError.ErrorDetail(errors.New("修改访问规则用户范围失败"))
			return
		}

		//新增规则用户范围
		_, err = session.Insert(&systemRuleUsers)
		if err != nil {
			conf.Logger.Error("修改访问规则用户范围失败", zap.Error(err))
			err = resp.DbInsertError.ErrorDetail(errors.New("修改访问规则用户范围失败"))
			return
		}
		return
	})
	if err != nil {
		return
	}
	return
}

// DelAccessRule 删除访问规则
func DelAccessRule(params []string) (err error) {
	db, err := client.GetDbClient()
	if err != nil {
		err = resp.DbConnectError.ErrorDetail(err)
		return
	}
	_, err = db.Transaction(func(session *xorm.Session) (data interface{}, err error) {
		for _, v := range params {
			//删除 规则记录
			_, err = session.Where("rule_id = ?", v).Delete(new(entity.SystemAccessRule))
			if err != nil {
				conf.Logger.Error("删除访问规则失败", zap.Error(err))
				err = resp.DbInsertError.ErrorDetail(errors.New("删除访问规则失败"))
				return
			}

			//删除 规则用户范围
			_, err = session.Where("rule_id = ?", v).Delete(new(entity.SystemRuleUser))
			if err != nil {
				conf.Logger.Error("删除访问规则用户范围失败", zap.Error(err))
				err = resp.DbInsertError.ErrorDetail(errors.New("删除访问规则用户范围失败"))
				return
			}
		}
		return
	})
	if err != nil {
		return
	}
	return
}

// ListAccessRule 访问规则列表
func ListAccessRule(params *request.ListAccessRuleReq) ([]response.SystemAccessRule, int64, error) {
	var systemAccessRule []response.SystemAccessRule
	db, err := client.GetDbClient()
	if err != nil {
		err = resp.DbConnectError.ErrorDetail(err)
		return nil, 0, err
	}

	modelObj := db.Table("system_access_rule as sar")
	modelObj.Join("left", "system_rule_user as sru", "sar.rule_id = sru.rule_id").
		Select("sar.ID,sar.rule_id,sar.rule_name,sar.rule_type,COUNT ( 1 ) AS num_user,sar.rule_detail,sar.STATE,sar.created_by,sar.created_time,sar.updated_by,sar.updated_time ")

	//条件查询
	if params.Search != "" {
		keyword := util.SpecialEscape(params.Search)
		modelObj.Where("sar.rule_name ilike ? or su.rule_detail ilike ?", "%"+keyword+"%", "%"+keyword+"%")
	}
	if params.UpdatedTimeFrom != "" {
		updatedTimeFrom, _ := time.Parse("2006-01-02 15:04:05", params.UpdatedTimeFrom)
		modelObj.Where("sar.updated_time > ?", updatedTimeFrom)
	}
	if params.UpdatedTimeTo != "" {
		updatedTimeTo, _ := time.Parse("2006-01-02 15:04:05", params.UpdatedTimeTo)
		modelObj.Where("sar.updated_time < ?", updatedTimeTo)
	}

	count, err := modelObj.GroupBy("sar.ID,sar.rule_id,sar.rule_name,sar.rule_type,sar.rule_detail,sar.STATE").
		OrderBy("sar.updated_time desc").Limit(params.PageSize, (params.Page-1)*params.PageSize).FindAndCount(&systemAccessRule)
	if err != nil {
		conf.Logger.Error("查询访问规则列表失败", zap.Error(err))
		err = resp.DbSelectError.ErrorDetail(errors.New("查询访问规则列表失败"))
		return nil, 0, err
	}

	return systemAccessRule, count, nil
}

// ListRuleUser 查询用户详情列表
func ListRuleUser(params *request.ListRuleUserReq) ([]response.SystemUserAccessRule, int64, error) {
	var systemUserAccessRule []response.SystemUserAccessRule
	db, err := client.GetDbClient()
	if err != nil {
		err = resp.DbConnectError.ErrorDetail(err)
		return nil, 0, err
	}

	modelObj := db.Table("system_access_rule as sar")
	modelObj.Join("INNER", "system_rule_user as sru", "sar.rule_id = sru.rule_id")

	if params.Disp == 0 {
		//展示已选用户
		modelObj.Join("INNER", "\"system_user\" as su", "sru.user_id = su.id").
			Join("INNER", "system_organization as so", "su.organization_id = so.organization_id").
			Select("su.system_account,su.phone,so.name,sru.user_id,0 as selected")

		modelObj.Where("sar.id = ?", params.Id)
	} else if params.Disp == 1 {
		//展示所有用户
		modelObj.Join("RIGHT", "\"system_user\" as su", "sru.user_id = su.id").
			Join("RIGHT", "system_organization as so", "su.organization_id = so.organization_id").
			Select("su.id as user_id,su.system_account,su.phone,so.name," + fmt.Sprintf("( CASE WHEN sar.ID = %v THEN 1 ELSE 0 END ) AS selected", params.Id))
	}

	modelObj.Where("so.is_deleted = 0 and su.is_deleted = 0")

	if params.Search != "" {
		keyword := util.SpecialEscape(params.Search)
		modelObj.Where("su.system_account like ? or su.phone like ?", "%"+keyword+"%", "%"+keyword+"%")
	}

	if params.OrganizationId != "" {
		modelObj.Where("so.organization_id = ?", params.OrganizationId)
	}

	count, err := modelObj.OrderBy("selected DESC,sar.updated_time").
		Limit(params.GetPageSize(), (params.GetPage()-1)*params.GetPageSize()).
		FindAndCount(&systemUserAccessRule)
	if err != nil {
		conf.Logger.Error("查询用户列表失败", zap.Error(err))
		err = resp.DbSelectError.ErrorDetail(errors.New("查询用户列表失败"))
		return nil, 0, err
	}

	return systemUserAccessRule, count, nil
}

func UpdateState(id, state int, updatedBy int) (err error) {

	db, err := client.GetDbClient()
	if err != nil {
		err = resp.DbConnectError.ErrorDetail(err)
		return
	}

	systemAccessRule := entity.SystemAccessRule{
		State:       state,
		UpdatedBy:   updatedBy,
		UpdatedTime: jsontime.Time(time.Now()),
	}

	_, err = db.Transaction(func(session *xorm.Session) (data interface{}, err error) {

		_, err = session.Where("id = ?", id).Cols("state,updated_by,updated_time").Update(&systemAccessRule)
		if err != nil {
			conf.Logger.Error("修改访问规则失败", zap.Error(err))
			err = resp.DbInsertError.ErrorDetail(errors.New("修改访问规则失败"))
			return
		}
		return
	})
	if err != nil {
		return
	}
	return
}
