package service

import (
	"crypto/md5"
	"encoding/hex"
	"errors"
	"fmt"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/bean/entity"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/bean/vo/request"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/bean/vo/response"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/common/client"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/common/conf"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/pkg/beagle/jsontime"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/pkg/beagle/resp"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/util"
	"time"
	"xorm.io/xorm"

	"go.uber.org/zap"
	"strings"
)

const (
	defaultPassword = "123456" // 重置默认密码
)

type User struct {
	User entity.SystemUserInfo
}

// 查询组织用户详情
func (o *User) OrgUserDetail(id int) (interface{}, error) {
	db, err := client.GetDbClient()
	if err != nil {
		return nil, resp.DbConnectError.WithError(err)
	}
	selectSql := "su.id, su.organization_id, su.phone, su.system_account, so.name as organization, su.created_by,u.name as created_name, " +
		"su.updated_time, su.created_time, su.name, su.logo, su.state"
	var result response.OrgUserDetail
	has, err := db.Table("system_user").Alias("su").
		Select(selectSql).
		Join("left", "system_organization so", "so.organization_id = su.organization_id").
		Join("left", "system_user u", "u.id = su.created_by").
		Where("su.id = ? and su.is_deleted = 0", id).Get(&result)
	if err != nil {
		conf.Logger.Error("查询组织用户详情失败", zap.Error(err))
		return nil, resp.DbSelectError.WithError(err)
	}
	if !has {
		return nil, resp.DbSelectError.ErrorDetail(errors.New("用户不存在"))
	}

	//查询用户角色
	var systemUserRoles []response.SystemUserRole
	if err := db.Table("system_user_role").Alias("sur").Select(" sr.role_name,sr.role_id").
		Join("left", "system_role sr", "sr.role_id = sur.role_id").Where("sur.user_id = ?", result.Id).Find(&systemUserRoles); err != nil {
		conf.Logger.Error("查询用户角色信息失败", zap.Error(err))
		return nil, resp.DbSelectError.ErrorDetail(err)
	}
	for _, v := range systemUserRoles {
		result.SystemRole = append(result.SystemRole, v.RoleId)
		result.SystemRoleName = append(result.SystemRoleName, v.RoleName)
	}
	return result, nil
}

// 组织添加用户时查询角色列表 查询非内置角色 + 平台运营者
func (o *User) OrgUserRoles(oid string) (interface{}, error) {
	db, err := client.GetDbClient()
	if err != nil {
		return nil, resp.DbConnectError.ErrorDetail(err)
	}

	var org entity.SystemOrganization
	_, err = db.Table("system_organization").Where("organization_id = ?", oid).Get(&org)
	if err != nil {
		conf.Logger.Error("查询机构信息失败", zap.Error(err))
		return nil, resp.DbSelectError.ErrorDetail(err)
	}
	session := db.Table("system_role").Where(" is_deleted = 0 and state = 1 ")
	if org.DataType == 2 {
		session.Where("role_type = 0 or role_type = 2")
	} else {
		session.Where("role_type = 0 or role_type = 1")
	}
	var roles []entity.SystemRole
	if err := session.Find(&roles); err != nil {
		conf.Logger.Error("查询用户角色信息失败", zap.Error(err))
		return nil, resp.DbSelectError.ErrorDetail(err)
	}

	return roles, nil
}

// 添加组织用户
func (o *User) OrgAddUser(input request.OrgUserInput) error {
	db, err := client.GetDbClient()
	if err != nil {
		return resp.DbConnectError.WithError(err)
	}

	//验证账号是否存在
	has, err := db.Table("system_user").Select("id").Where("system_account = ? and is_deleted = 0", input.SystemAccount).Exist(&entity.SystemUser{})
	if err != nil {
		conf.Logger.Error("查询账号信息验证失败", zap.Error(err))
		return resp.DbInsertError.WithError(err)
	}
	if has {
		return resp.DbInsertError.WithMsg("账号已存在")
	}

	//验证手机号是否存在
	has, err = db.Table("system_user").Select("id").Where("phone = ? and is_deleted = 0", input.Phone).Exist(&entity.SystemUser{})
	if err != nil {
		conf.Logger.Error("查询账号信息验证失败", zap.Error(err))
		return resp.DbInsertError.ErrorDetail(err)
	}
	if has {
		return resp.DbInsertError.WithMsg("手机号已存在")
	}

	userModel := entity.SystemUser{
		OrganizationId: input.OrganizationId,
		Name:           input.Name,
		Logo:           input.Logo,
		SystemAccount:  input.SystemAccount,
		Phone:          input.Phone,
		State:          input.State,
		CreatedBy:      o.User.Id,
	}

	_, err = db.Transaction(func(s *xorm.Session) (interface{}, error) {
		if _, err := s.Table("system_user").Insert(&userModel); err != nil {
			conf.Logger.Error("创建系统账户失败", zap.Error(err))
			return nil, resp.DbInsertError.ErrorDetail(err)
		}

		upperMd5Pass, err := SolvePassword(userModel.Id, input.Password)
		if err != nil {
			conf.Logger.Error("密码处理出错", zap.Error(err))
			return nil, resp.DbInsertError.ErrorDetail(err)
		}
		_, err = s.Table("system_user").Where("id = ?", userModel.Id).Update(map[string]string{
			"password": upperMd5Pass,
		})
		if err != nil {
			conf.Logger.Error("密码处理出错", zap.Error(err))
			return nil, resp.DbInsertError.ErrorDetail(err)
		}

		var systemUserRoles []entity.SystemUserRole
		for _, v := range input.SelectRole {
			systemUserRoles = append(systemUserRoles, entity.SystemUserRole{UserId: userModel.Id, RoleId: v})
		}
		if _, err := s.Table("system_user_role").Insert(&systemUserRoles); err != nil {
			conf.Logger.Error("创建用户角色失败", zap.Error(err))
			return nil, resp.DbInsertError.ErrorDetail(err)
		}
		return nil, nil
	})
	return err
}

// 更新组织用户
func (o *User) OrgUpdateUser(id int, input request.UpdateOrgUserInput) error {
	db, err := client.GetDbClient()
	if err != nil {
		return resp.DbConnectError.ErrorDetail(err)
	}
	var baseUser entity.SystemUser
	has, err := db.Table("system_user").ID(id).Where("is_deleted = 0").Get(&baseUser)
	if err != nil {
		conf.Logger.Error("查询账号信息失败", zap.Error(err))
		return resp.DbUpdateError.ErrorDetail(err)
	}
	if !has {
		return resp.DbUpdateError.ErrorDetail(errors.New("该用户不存在"))
	}

	if baseUser.SystemAccount != input.SystemAccount {
		//验证账号是否存在
		has, err = db.Table("system_user").Select("id").Where("system_account = ? and is_deleted = 0", input.SystemAccount).Exist(&entity.SystemUser{})
		if err != nil {
			conf.Logger.Error("查询账号信息验证失败", zap.Error(err))
			return resp.DbUpdateError.ErrorDetail(err)
		}
		if has {
			return resp.DbUpdateError.ErrorDetail(errors.New("账号已存在"))
		}
	}

	if baseUser.Phone != input.Phone {
		//验证手机号是否存在
		has, err = db.Table("system_user").Select("id").Where("phone = ? and is_deleted = 0", input.Phone).Exist(&entity.SystemUser{})
		if err != nil {
			conf.Logger.Error("查询账号信息验证失败", zap.Error(err))
			return resp.DbUpdateError.ErrorDetail(err)
		}
		if has {
			return resp.DbUpdateError.ErrorDetail(errors.New("手机号已存在"))
		}
	}

	userModel := entity.SystemUser{
		Logo:          input.Logo,
		SystemAccount: input.SystemAccount,
		Phone:         input.Phone,
		State:         input.State,
		Name:          input.Name,
		UpdatedBy:     o.User.Id,
	}

	if _, err := db.Table("system_user").ID(id).Cols("logo,system_account,phone,state,name,update_by").Update(&userModel); err != nil {
		conf.Logger.Error("账号信息更新失败", zap.Error(err))
		return resp.DbUpdateError.ErrorDetail(err)
	}
	return nil
}

// 删除组织用户
func (o *User) DelOrgUser(input request.DelOrgUser) error {
	db, err := client.GetDbClient()
	if err != nil {
		return resp.DbConnectError.ErrorDetail(err)
	}

	_, err = db.Transaction(func(s *xorm.Session) (interface{}, error) {
		var baseUsers []entity.SystemUser
		if err := s.Table("system_user").In("id", input.Ids).Where("is_deleted = 0").Find(&baseUsers); err != nil {
			conf.Logger.Error("查询账号信息失败", zap.Error(err))
			return nil, resp.DbDeleteError.ErrorDetail(err)
		}
		if len(baseUsers) == 0 {
			return nil, resp.DbDeleteError.ErrorDetail(errors.New("用户数据不存在，删除失败"))
		}
		for _, v := range baseUsers {
			if v.State == 1 {
				return nil, resp.DbDeleteError.ErrorDetail(errors.New("删除内容中有启用状态的数据，不可删除！"))
			}
		}

		if _, err := db.Table("system_user").In("id", input.Ids).Update(&entity.SystemUser{IsDeleted: 1, UpdatedBy: o.User.Id}); err != nil {
			conf.Logger.Error("用户删除失败", zap.Error(err))
			return nil, resp.DbDeleteError.ErrorDetail(err)
		}

		//删除用户关联角色
		var user_ids []int
		for _, v := range baseUsers {
			user_ids = append(user_ids, v.Id)
		}
		if _, err := db.Table("system_user_role").In("user_id", user_ids).Delete(&entity.SystemUserRole{}); err != nil {
			conf.Logger.Error("删除用户关联角色失败", zap.Error(err))
			return nil, resp.DbDeleteError.ErrorDetail(err)
		}
		return nil, nil
	})
	return err
}

// 去重校验
func (o *User) CheckRepetition(input request.CheckRepetition) error {
	db, err := client.GetDbClient()
	if err != nil {
		return resp.DbConnectError.ErrorDetail(err)
	}

	var system_user entity.SystemUser
	if input.Id != 0 {
		if _, err := db.Table("system_user").Select("id, phone").ID(input.Id).Where("is_deleted = 0").Get(&system_user); err != nil {
			conf.Logger.Error("查询用户失败", zap.Error(err))
			return resp.DbDataCheckError.ErrorDetail(err)
		}
	}
	if input.Phone != system_user.Phone {
		has, err := db.Table("system_user").Select("id").Where("phone = ? and is_deleted = 0", &input).Exist(&entity.SystemUser{})
		if err != nil {
			conf.Logger.Error("手机号去重查询失败", zap.Error(err))
			return resp.DbDataCheckError.ErrorDetail(err)
		}
		if has {
			return resp.DbDataCheckError.ErrorDetail(errors.New("该手机号已注册"))
		}
	}
	return nil
}

func (u *User) SystemUserUpdateState(params request.SystemUserStateReq) error {
	db, err := client.GetDbClient()
	if err != nil {
		return resp.DbConnectError.WithData(err)
	}

	_, err = db.Table("system_user").Cols("state,updated_by").Where("id = ? ", params.Id).Update(map[string]interface{}{
		"updated_by":   u.User.Id,
		"state":        params.State,
		"updated_time": jsontime.Time(time.Now()),
	})
	if err != nil {
		return resp.DbSelectError.WithData(err)
	}

	return nil
}

func (o *User) DevOps() ([]entity.SystemUserInfo, error) {
	var (
		err    error
		result []entity.SystemUserInfo
	)

	db, err := client.GetDbClient()
	if err != nil {
		return result, resp.DbConnectError.WithError(err)
	}

	err = db.Table("system_user").Alias("su").
		Join("LEFT", "system_user_role sur", "su.id = sur.user_id").
		Join("LEFT", "system_role sr", "sur.role_id = sr.role_id").
		Where("su.is_deleted = 0 and su.state =1 and sr.is_deleted = 0 and sr.state = 1 and role_type = 1").
		Select("DISTINCT su.*").Find(&result)
	if err != nil {
		return result, resp.DbSelectError.WithError(err)
	}

	return result, err

}

// SystemUserEditPassword 修改密码
func SystemUserEditPassword(params request.SystemUserEditPasswordReq) (err error) {
	db, err := client.GetDbClient()
	if err != nil {
		err = resp.DbConnectError.ErrorDetail(err)
		return
	}
	oldSystemUser := entity.SystemUser{}
	has, err := db.Cols("password", "id").ID(params.Id).Get(&oldSystemUser)
	if err != nil || !has {
		conf.Logger.Error("查询系统账户详情失败", zap.Error(err))
		err = resp.DbSelectError.ErrorDetail(errors.New("查询系统账户详情失败"))
		return
	}

	password, err := SolvePassword(oldSystemUser.Id, params.Password)
	if err != nil {
		return
	}
	SystemUser := entity.SystemUser{}
	SystemUser.Password = password

	_, err = db.Where("id=?", oldSystemUser.Id).Where("is_deleted = 0").Update(&SystemUser)
	if err != nil {
		conf.Logger.Error("修改系统账户失败", zap.Error(err))
		err = resp.DbUpdateError.ErrorDetail(errors.New("修改系统账户失败"))
		return
	}
	return
}

// ResetSystemUserPassword 批量重置密码
func ResetSystemUserPassword(params request.ResetSystemUserPasswordReq) (err error) {
	db, err := client.GetDbClient()
	if err != nil {
		err = resp.DbConnectError.ErrorDetail(err)
		return
	}

	str, EncryptErr := util.EncryptPwd(defaultPassword)
	if EncryptErr != nil {
		err = resp.DbUpdateError.ErrorDetail(err)
	}
	for _, v := range params.Ids {
		password, SolvePwdErr := SolvePassword(v, str)
		if SolvePwdErr != nil {
			return SolvePwdErr
		}
		updateData := entity.SystemUser{}
		updateData.Password = password
		_, err = db.Where("id=?", v).Update(&updateData)
		if err != nil {
			conf.Logger.Error("重置账户密码失败", zap.Error(err))
			err = resp.DbUpdateError.ErrorDetail(errors.New("重置账户密码失败"))
			return
		}
	}
	return
}

// SolvePassword 密码处理
func SolvePassword(userId int, password string) (string, error) {
	passwordKey, err := util.DecryptPwd(password)
	if passwordKey == "" {
		return "", resp.InvalidParam.WithError(err)
	}
	h := md5.New()
	_, err = h.Write([]byte(strings.ToUpper(fmt.Sprintf("%v-%v", userId, passwordKey))))
	if err != nil {
		return "", resp.InvalidParam.WithError(err)
	}
	md5Pwd := strings.ToUpper(hex.EncodeToString(h.Sum(nil)))
	return md5Pwd, nil
}
