package service

import (
	"errors"
	"fmt"
	json "github.com/json-iterator/go"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/bean/entity"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/bean/vo/request"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/bean/vo/response"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/common/client"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/pkg/beagle/resp"
	"time"
)

type WorkOrderManageSvc struct {
	User *entity.SystemUser
}

// AddWorkOrderManage 新增业务工单
func (w *WorkOrderManageSvc) AddWorkOrderManage(req request.AddWorkOrderReq) (err error) {
	var timingWeekly, timingCustom, pushObj []byte
	db, err := client.GetDbClient()
	if err != nil {
		err = resp.DbConnectError.WithError(err)
		return
	}

	// 校验 工单名称 是否重复
	has, err := db.Table("work_order_manage").Where("order_name = ?", req.OrderName).Exist()
	if has {
		err = resp.DbDataCheckError.WithError(errors.New("工单名称重复"))
		return
	}
	if err != nil {
		err = resp.DbSelectError.WithError(err)
		return
	}

	pushObj, err = json.Marshal(req.PushObj)
	if err != nil {
		err = resp.FAIL.WithError(err)
		return
	}

	workOrder := entity.WorkOrder{
		OrderName:  req.OrderName,
		OrderLevel: req.OrderLevel,
		OrderDesc:  req.OrderDesc,
		PushObj:    fmt.Sprintf("%s", pushObj),
		TimingType: req.TimingType,
		CreateUser: "",
		CreateTime: time.Now(),
		UpdateUser: "",
		UpdateTime: time.Now(),
	}

	if req.TimingType == 2 {
		timingWeekly, err = json.Marshal(req.TimingWeekly)
		if err != nil {
			err = resp.FAIL.WithError(err)
			return
		}
		workOrder.TimingRule = fmt.Sprintf("%s", timingWeekly)
	} else if req.TimingType == 3 {
		timingCustom, err = json.Marshal(req.TimingCustom)
		if err != nil {
			err = resp.FAIL.WithError(err)
			return
		}
		workOrder.TimingRule = fmt.Sprintf("%s", timingCustom)
	}

	_, err = db.Table("work_order_manage").Insert(&workOrder)
	if err != nil {
		err = resp.DbInsertError.WithError(err)
		return
	}

	//是否立刻下发
	if req.IsPush == 1 {
		push := request.PushWorkOrderReq{
			Id:      workOrder.Id,
			PushObj: req.PushObj,
		}
		w.PushWorkOrderManage(push)
	}

	return
}

// EditWorkOrderManage 编辑业务工单管理
func (w *WorkOrderManageSvc) EditWorkOrderManage(req request.EditWorkOrderReq) (err error) {
	var timingWeekly, timingCustom, pushObj []byte
	db, err := client.GetDbClient()
	if err != nil {
		err = resp.DbConnectError.WithError(err)
		return
	}

	pushObj, err = json.Marshal(req.PushObj)
	if err != nil {
		err = resp.FAIL.WithError(err)
		return
	}

	workOrder := entity.WorkOrder{
		OrderLevel: req.OrderLevel,
		OrderDesc:  req.OrderDesc,
		PushObj:    fmt.Sprintf("%s", pushObj),
		TimingType: req.TimingType,
		UpdateUser: "",
		UpdateTime: time.Now(),
	}
	if req.TimingType == 2 {
		timingWeekly, err = json.Marshal(req.TimingWeekly)
		if err != nil {
			err = resp.FAIL.WithError(err)
			return
		}
		workOrder.TimingRule = fmt.Sprintf("%s", timingWeekly)
	} else if req.TimingType == 3 {
		timingCustom, err = json.Marshal(req.TimingCustom)
		if err != nil {
			err = resp.FAIL.WithError(err)
			return
		}
		workOrder.TimingRule = fmt.Sprintf("%s", timingCustom)
	}

	_, err = db.Table("work_order_manage").Where("id = ?", req.Id).
		Cols("order_level,order_desc,order_cnt,push_obj,timing_type,timing_rule").Update(&workOrder)
	if err != nil {
		err = resp.DbUpdateError.WithError(err)
		return
	}

	//是否立刻下发
	if req.IsPush == 1 {
		push := request.PushWorkOrderReq{
			Id:      req.Id,
			PushObj: req.PushObj,
		}
		w.PushWorkOrderManage(push)
	}
	return
}

// StateWorkOrderManage 编辑业务工单管理
func (w *WorkOrderManageSvc) StateWorkOrderManage(req request.StateWorkOrderReq) (err error) {
	db, err := client.GetDbClient()
	if err != nil {
		err = resp.DbConnectError.WithError(err)
		return
	}

	_, err = db.Table("work_order_manage").Where("id = ?", req.Id).
		Cols("timing_state").Update(&entity.WorkOrder{TimingState: req.TimingState})
	if err != nil {
		err = resp.DbUpdateError.WithError(err)
		return
	}
	return
}

// DelWorkOrderManage 删除
func (w *WorkOrderManageSvc) DelWorkOrderManage(req request.DelWorkOrderReq) (err error) {
	db, err := client.GetDbClient()
	if err != nil {
		err = resp.DbConnectError.WithError(err)
		return
	}

	_, err = db.Table("work_order_manage").In("id", req.Id).Delete(&entity.WorkOrder{})
	if err != nil {
		err = resp.DbDeleteError.WithError(err)
		return
	}
	return
}

// DetailsWorkOrderManage 详情
func (w *WorkOrderManageSvc) DetailsWorkOrderManage(id int) (workOrderRes response.WorkOrderRes, err error) {
	db, err := client.GetDbClient()
	if err != nil {
		err = resp.DbConnectError.WithError(err)
		return
	}

	//查询任务详情
	var workOrder entity.WorkOrder
	finder := db.Table("work_order_manage").Where("id = ?", id)
	has, err := finder.Get(&workOrder)
	if err != nil {
		err = resp.DbSelectError.WithError(err)
		return
	}
	if !has {
		err = resp.DbSelectError.WithError(errors.New("查询为空"))
		return
	}

	err = json.Unmarshal([]byte(workOrder.PushObj), &workOrderRes.PushObj)
	if err != nil {
		err = resp.FAIL.WithError(err)
		return
	}
	if workOrder.TimingType == 2 {
		var timingWeekly request.TimingWeekly
		err = json.Unmarshal([]byte(workOrder.TimingRule), &timingWeekly)
		if err != nil {
			err = resp.FAIL.WithError(err)
			return
		}
		workOrderRes.TimingRule = timingWeekly
	} else if workOrder.TimingType == 3 {
		var timingCustom []request.TimingCustom
		err = json.Unmarshal([]byte(workOrder.TimingRule), &timingCustom)
		if err != nil {
			err = resp.FAIL.WithError(err)
			return
		}
		workOrderRes.TimingRule = timingCustom
	}

	workOrderRes.Id = workOrder.Id
	workOrderRes.OrderName = workOrder.OrderName
	workOrderRes.OrderLevel = workOrder.OrderLevel
	workOrderRes.OrderDesc = workOrder.OrderDesc
	workOrderRes.TimingState = workOrder.TimingState
	workOrderRes.TimingType = workOrder.TimingType
	workOrderRes.CreateUser = workOrder.CreateUser
	workOrderRes.CreateTime = workOrder.CreateTime
	workOrderRes.UpdateUser = workOrder.UpdateUser
	workOrderRes.UpdateUser = workOrder.UpdateUser
	workOrderRes.UpdateTime = workOrder.UpdateTime

	return
}

// ListWorkOrderManage 列表
func (w *WorkOrderManageSvc) ListWorkOrderManage(req request.ListWorkOrderManageReq) (total int64, workOrderListRes []response.WorkOrderListRes, err error) {
	db, err := client.GetDbClient()
	if err != nil {
		err = resp.DbConnectError.WithError(err)
		return
	}

	finder := db.Table("work_order_manage")
	if req.Search != "" {
		finder.Where(fmt.Sprintf("order_name LIKE '%s'", "%"+req.Search+"%"))
	}
	if req.OrderLevel != 0 {
		finder.Where("order_level = ?", req.OrderLevel)
	}
	if req.TimingType != 0 {
		finder.Where("timing_type = ?", req.TimingType)
	}
	if req.CreateDateFrom != "" {
		finder.Where("create_time >= ?", req.CreateDateFrom)
	}
	if req.CreateDateTo != "" {
		finder.Where("create_time <= ?", req.CreateDateTo)
	}
	finder.OrderBy("create_time")
	//查询任务
	total, err = finder.Select("id,order_name,order_level,order_cnt,push_obj,timing_type,timing_state,create_user,create_time").
		Limit(req.PageSize, (req.Page-1)*req.PageSize).FindAndCount(&workOrderListRes)
	if err != nil {
		err = resp.DbSelectError.WithError(err)
		return
	}

	for k, v := range workOrderListRes {
		var pushObj request.PushObj
		err = json.Unmarshal([]byte(v.PushObj), &pushObj)
		if err != nil {
			err = resp.FAIL.WithError(err)
			return
		}
		workOrderListRes[k].PushObj = pushObj.UserObj[0].UserName
		for i := 1; i < len(pushObj.UserObj); i++ {
			workOrderListRes[k].PushObj = workOrderListRes[k].PushObj + "、" + pushObj.UserObj[i].UserName
		}
	}
	return
}

// PushWorkOrderManage 手动下发
func (w *WorkOrderManageSvc) PushWorkOrderManage(req request.PushWorkOrderReq) (err error) {
	db, err := client.GetDbClient()
	if err != nil {
		err = resp.DbConnectError.WithError(err)
		return
	}

	session := db.NewSession()
	defer session.Close()
	session.Begin()

	//pushObj := req.PushObj.UserObj[0].UserName
	//for i := 1; i < len(req.PushObj.UserObj); i++ {
	//	pushObj = pushObj + "、" + req.PushObj.UserObj[i].UserName
	//}
	pushObj, err := json.Marshal(req.PushObj)
	if err != nil {
		err = resp.FAIL.WithError(err)
		return
	}

	//新增 下发工单表 数据
	workOrderIssuance := entity.WorkOrderIssuance{
		OrderId:    req.Id,
		PushObj:    fmt.Sprintf("%s", pushObj),
		OrderState: 1,
		CreateUser: "admin",
		CreateTime: time.Now(),
	}
	_, err = session.Table("work_order_issuance").Insert(&workOrderIssuance)
	if err != nil {
		err = resp.DbInsertError.WithError(err)
		session.Rollback()
		return
	}
	for _, v := range req.PushObj.UserObj {
		//新增 个人业务工单表 数据
		workOrderMe := entity.WorkOrderMe{
			OrderId:         req.Id,
			OrderIssuanceId: workOrderIssuance.Id,
			SystemAccount:   v.SystemAccount,
			OrderState:      1,
		}
		_, err = session.Table("work_order_me").Insert(&workOrderMe)
		if err != nil {
			err = resp.DbInsertError.WithError(err)
			session.Rollback()
			return
		}
	}

	//发消息通知 czl..

	//修改库表实例工单数
	_, err = session.Table("work_order_manage").Where("id = ?", req.Id).Incr("order_cnt").Update(&entity.WorkOrder{})
	if err != nil {
		err = resp.DbUpdateError.WithError(err)
		session.Rollback()
		return
	}
	session.Commit()
	return
}

// CloseWorkOrderIssuance 关闭工单
func (w *WorkOrderManageSvc) CloseWorkOrderIssuance(req request.CloseWorkOrderReq) (err error) {
	db, err := client.GetDbClient()
	if err != nil {
		err = resp.DbConnectError.WithError(err)
		return
	}

	session := db.NewSession()
	defer session.Close()
	session.Begin()

	//修改 下发工单表 数据
	_, err = session.Table("work_order_issuance").Where("id = ?", req.Id).Cols("order_state,reason_closure").
		Update(&entity.WorkOrderIssuance{
			OrderState:    3,
			ReasonClosure: req.ReasonClosure,
		})
	if err != nil {
		err = resp.DbInsertError.WithError(err)
		session.Rollback()
		return
	}
	//修改 个人业务工单表 数据
	_, err = session.Table("work_order_me").Where("order_issuance_id = ?", req.Id).Cols("order_state").
		Update(&entity.WorkOrderMe{
			OrderState: 3,
		})
	if err != nil {
		err = resp.DbInsertError.WithError(err)
		session.Rollback()
		return
	}

	session.Commit()
	return
}

// ListWorkOrderIssuance 业务工单下发列表
func (w *WorkOrderManageSvc) ListWorkOrderIssuance(req request.ListWorkOrderReq) (total int64, workOrderIssuanceListRes []response.WorkOrderIssuanceListRes, err error) {
	db, err := client.GetDbClient()
	if err != nil {
		err = resp.DbConnectError.WithError(err)
		return
	}

	finder := db.Table("work_order_issuance").Alias("woi").
		Join("INNER", "work_order_manage wom", "woi.order_id = wom.id")

	if req.Search != "" {
		finder.Where(fmt.Sprintf("wom.order_name LIKE '%s'", "%"+req.Search+"%"))
	}
	if req.OrderLevel != 0 {
		finder.Where("wom.order_level = ?", req.OrderLevel)
	}
	if req.OrderState != 0 {
		finder.Where("woi.order_state = ?", req.OrderState)
	}
	if req.CreateDateFrom != "" {
		finder.Where("woi.create_time >= ?", req.CreateDateFrom)
	}
	if req.CreateDateTo != "" {
		finder.Where("woi.create_time <= ?", req.CreateDateTo)
	}
	if req.CompleteTimeFrom != "" {
		finder.Where("woi.complete_time >= ?", req.CompleteTimeFrom)
	}
	if req.CompleteTimeTo != "" {
		finder.Where("woi.complete_time <= ?", req.CompleteTimeTo)
	}
	if req.Id != 0 {
		finder.Where("wom.id = ?", req.Id)
	}
	finder.OrderBy("woi.create_time")
	//查询任务
	total, err = finder.Select("woi.\"id\",woi.order_id,wom.order_name,wom.order_level,woi.order_state,woi.push_obj,woi.create_user,woi.create_time,woi.complete_time,"+
		"(select count(1) from work_order_me wome where wome.order_issuance_id = woi.id and wome.order_state = woi.order_state) as state_cnt,"+
		"(select count(1) from work_order_me wome where wome.order_issuance_id = woi.id) as all_cnt").
		Limit(req.PageSize, (req.Page-1)*req.PageSize).FindAndCount(&workOrderIssuanceListRes)
	if err != nil {
		err = resp.DbSelectError.WithError(err)
		return
	}

	for k, v := range workOrderIssuanceListRes {
		var pushObj request.PushObj
		err = json.Unmarshal([]byte(v.PushObj), &pushObj)
		if err != nil {
			err = resp.FAIL.WithError(err)
			return
		}
		workOrderIssuanceListRes[k].PushObj = pushObj.UserObj[0].UserName
		for i := 1; i < len(pushObj.UserObj); i++ {
			workOrderIssuanceListRes[k].PushObj = workOrderIssuanceListRes[k].PushObj + "、" + pushObj.UserObj[i].UserName
		}
	}
	return
}

// DetailsWorkOrderIssuance 业务工单下发详情
func (w *WorkOrderManageSvc) DetailsWorkOrderIssuance(id int) (workOrderIssuanceRes response.WorkOrderIssuanceRes, err error) {
	db, err := client.GetDbClient()
	if err != nil {
		err = resp.DbConnectError.WithError(err)
		return
	}

	var workOrderIssuance response.WorkOrderIssuance
	finder := db.Table("work_order_issuance").Alias("woi").
		Join("INNER", "work_order_manage wom", "woi.order_id = wom.ID").Where("woi.id = ?", id)
	has, err := finder.Select("woi.\"id\",woi.order_id,wom.order_name,wom.order_level,wom.order_desc,woi.order_state,woi.push_obj,woi.create_user,woi.create_time,woi.complete_time").
		Get(&workOrderIssuance)
	if err != nil {
		err = resp.DbSelectError.WithError(err)
		return
	}
	if !has {
		return
	}

	//var workOrderMe entity.WorkOrderMe
	finder1 := db.Table("work_order_me").Where("order_issuance_id = ?", id)
	err = finder1.Find(&workOrderIssuanceRes.WorkOrderMe)
	if err != nil {
		err = resp.DbSelectError.WithError(err)
		return
	}

	err = json.Unmarshal([]byte(workOrderIssuance.PushObj), &workOrderIssuanceRes.PushObj)
	if err != nil {
		err = resp.FAIL.WithError(err)
		return
	}

	workOrderIssuanceRes.Id = workOrderIssuance.Id
	workOrderIssuanceRes.OrderId = workOrderIssuance.OrderId
	workOrderIssuanceRes.OrderName = workOrderIssuance.OrderName
	workOrderIssuanceRes.OrderLevel = workOrderIssuance.OrderLevel
	workOrderIssuanceRes.OrderDesc = workOrderIssuance.OrderDesc
	workOrderIssuanceRes.OrderState = workOrderIssuance.OrderState
	workOrderIssuanceRes.CreateUser = workOrderIssuance.CreateUser
	workOrderIssuanceRes.CreateTime = workOrderIssuance.CreateTime
	workOrderIssuanceRes.CompleteTime = workOrderIssuance.CompleteTime

	return
}

// FeedbackWorkOrderMe 处置反馈
func (w *WorkOrderManageSvc) FeedbackWorkOrderMe(req request.FeedbackWorkOrderReq) (err error) {
	db, err := client.GetDbClient()
	if err != nil {
		err = resp.DbConnectError.WithError(err)
		return
	}

	session := db.NewSession()
	defer session.Close()
	session.Begin()
	completeTime := time.Now()
	//修改 个人业务工单表 数据
	_, err = session.Table("work_order_me").Where("id = ?", req.Id).Cols("order_state,result_feedback,complete_time").
		Update(&entity.WorkOrderMe{
			OrderState:     2,
			ResultFeedback: req.ResultFeedback,
			CompleteTime:   completeTime,
		})
	if err != nil {
		err = resp.DbInsertError.WithError(err)
		session.Rollback()
		return
	}

	//查询此工单是否已完成
	has, err := session.Table("work_order_me").Where("order_state = 1 AND order_issuance_id = ?", req.OrderIssuanceId).Exist()
	if !has {
		//修改 下发工单 数据
		_, err = session.Table("work_order_issuance").Where("id = ?", req.OrderIssuanceId).Cols("order_state,complete_time").
			Update(&entity.WorkOrderIssuance{
				OrderState:   2,
				CompleteTime: completeTime,
			})
		if err != nil {
			err = resp.DbInsertError.WithError(err)
			session.Rollback()
			return
		}
	}
	if err != nil {
		err = resp.DbSelectError.WithError(err)
		session.Rollback()
		return
	}

	session.Commit()
	return
}

// ListWorkOrderMe 我的业务工单列表
func (w *WorkOrderManageSvc) ListWorkOrderMe(req request.ListWorkOrderReq) (total int64, workOrderMeListRes []response.WorkOrderMeListRes, err error) {
	db, err := client.GetDbClient()
	if err != nil {
		err = resp.DbConnectError.WithError(err)
		return
	}

	finder := db.Table("work_order_me").Alias("wome").
		Join("INNER", "work_order_issuance woi", "wome.order_issuance_id = woi.id").
		Join("INNER", "work_order_manage wom", "woi.order_id = wom.id")

	if req.Search != "" {
		finder.Where(fmt.Sprintf("wom.order_name LIKE '%s'", "%"+req.Search+"%"))
	}
	if req.OrderLevel != 0 {
		finder.Where("wom.order_level = ?", req.OrderLevel)
	}
	if req.OrderState != 0 {
		finder.Where("wome.order_state = ?", req.OrderState)
	}
	if req.CreateDateFrom != "" {
		finder.Where("woi.create_time >= ?", req.CreateDateFrom)
	}
	if req.CreateDateTo != "" {
		finder.Where("woi.create_time <= ?", req.CreateDateTo)
	}
	if req.CompleteTimeFrom != "" {
		finder.Where("wome.complete_time >= ?", req.CompleteTimeFrom)
	}
	if req.CompleteTimeTo != "" {
		finder.Where("wome.complete_time <= ?", req.CompleteTimeTo)
	}
	finder.OrderBy("woi.create_time")
	//查询任务
	total, err = finder.Select("wome.\"id\",wome.order_id,wome.order_issuance_id,wom.order_name,wom.order_level,"+
		"wome.order_state,woi.push_obj,woi.create_user,woi.create_time,wome.complete_time").
		Limit(req.PageSize, (req.Page-1)*req.PageSize).FindAndCount(&workOrderMeListRes)
	if err != nil {
		err = resp.DbSelectError.WithError(err)
		return
	}

	for k, v := range workOrderMeListRes {
		var pushObj request.PushObj
		err = json.Unmarshal([]byte(v.PushObj), &pushObj)
		if err != nil {
			err = resp.FAIL.WithError(err)
			return
		}
		workOrderMeListRes[k].PushObj = pushObj.UserObj[0].UserName
		for i := 1; i < len(pushObj.UserObj); i++ {
			workOrderMeListRes[k].PushObj = workOrderMeListRes[k].PushObj + "、" + pushObj.UserObj[i].UserName
		}
	}
	return
}
