package controller

import (
	"errors"
	vd "github.com/bytedance/go-tagexpr/validator"
	"github.com/gin-gonic/gin"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/bean/vo/request"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/common/conf"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/pkg/beagle/constant"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/pkg/beagle/resp"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/router/middleware/header"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/service"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/util"
	"go.uber.org/zap"
	"strconv"
	"time"
)

func UserLogin(c *gin.Context) {
	userReq := request.UserReq{}
	err := c.BindJSON(&userReq)
	if err != nil {
		SendJsonResponse(c, resp.InvalidParam.WithError(err), nil)
		return
	}
	//参数检测
	if err := vd.Validate(userReq); err != nil {
		SendJsonResponse(c, resp.InvalidParam.WithError(err), "")
		return
	}
	// 验证 userName 是否含有危险字符
	if util.IfDangerCharacter(userReq.SystemAccount) {
		SendJsonResponse(c, resp.InvalidParam.WithMsg("账号存在危险字符"), "")
		return
	}
	//TODO 解密password
	pwd, err := util.DecryptPwd(userReq.Password)
	if err != nil {
		SendJsonResponse(c, resp.FAIL.WithError(err), "")
		return
	}
	userReq.Password = pwd
	loginInf := service.UserSvc{SystemAccount: userReq.SystemAccount, PassWord: userReq.Password}
	msg, uuidStr, lastLogin, err := loginInf.Login()
	if err != nil {
		SendJsonResponse(c, err, "")
		return
	}
	c.SetCookie(conf.CookieName, uuidStr, 1*60*60*24, "/", "", false, false)
	c.SetCookie(conf.CookieNameLastLogin, lastLogin, 1*60*60*24, "/", "", false, false)
	SendJsonResponse(c, resp.OK, msg)
}

func UserLoginV2(c *gin.Context) {
	userReq := request.UserReq{}
	err := c.BindJSON(&userReq)
	if err != nil {
		SendJsonResponse(c, resp.InvalidParam.WithError(err), nil)
		return
	}
	// 参数检测
	if err = vd.Validate(userReq); err != nil {
		SendJsonResponse(c, resp.InvalidParam.WithError(err), nil)
		return
	}

	loginInf := service.UserSvc{SystemAccount: userReq.SystemAccount}
	userInfo, err := loginInf.GetUserInfo("")
	if err != nil {
		SendJsonResponse(c, err, nil)
		return
	}

	// 获取系统配置信息
	op := service.SystemOptionsSvc{}
	config, err := op.GetSystemOptions()
	if err != nil {
		SendJsonResponse(c, err, nil)
		return
	}
	// 未开启是否启用登录配置时 只进行密码正确性校验 (admin 始终只进行密码校验)

	upperMd5Pass, err := service.SolvePassword(userInfo.Id, userReq.Password)
	if err != nil {
		conf.Logger.Error("加密错误", zap.Error(err))
		SendJsonResponse(c, err, nil)
		return
	}
	if upperMd5Pass != userInfo.Password {
		if config.LoginConfigState == 1 {
			err := loginInf.UserLock(userInfo.Id, userInfo.Password, config)
			if err != nil {
				SendJsonResponse(c, err, nil)
				return
			}
		}
		err = errors.New("密码错误")
		SendJsonResponse(c, err, nil)
		return
	}
	loginInf.UserUnLock(userInfo.Id)

	// TODO STEP2 用户和ip没有加入到访问规则登录不上提醒
	if config.AccessRuleState == 1 {
		ip := util.RemoteIp(c.Request)
		svc := service.AccessRuleSvc{}
		if errCode := svc.CheckIp(ip, userInfo.Id); errCode != nil {
			SendJsonResponse(c, errCode, nil)
			return
		}
	}
	//  密码失效提醒
	subDays := util.SubDays(time.Now(), userInfo.PwdUpdatedTime)
	if config.PwdValidity-subDays <= 0 && config.PwdValidity != 0 && userInfo.PwdUpdatedTime.String()[:10] != "0001-01-01" {
		SendJsonResponse(c, resp.FAIL.ErrorDetail(errors.New("密码已失效，请联系管理员修改密码！")), nil)
		return
	}
	//  密码强弱提醒
	if config.ForceUpdateState == 1 && config.MinPwdLevel > userInfo.PwdLevel {
		SendJsonResponse(c, resp.FAIL.ErrorDetail(errors.New("密码强度弱，请联系管理员修改密码！")), nil)
		return
	}

	lastLogin, msg, uuidStr, err := loginInf.LoginV2(userInfo)
	if err != nil {
		SendJsonResponse(c, err, "")
		return
	}
	c.SetCookie(conf.CookieName, uuidStr, 1*60*60*24, "/", "", false, false)
	c.SetCookie(conf.CookieNameLastLogin, lastLogin, 1*60*60*24, "/", "", false, false)

	SendJsonResponse(c, resp.OK, msg)
}

func GetUserInfo(c *gin.Context) {

	token, _ := c.Cookie(conf.CookieName)
	conf.Logger.Info("当前token信息为", zap.String("bgToken", token))
	header := c.GetHeader(conf.CookieName)
	conf.Logger.Info("当前header信息为", zap.String("header", header))
	if token == "" {
		SendJsonResponse(c, resp.FAIL.WithMsg("登录超时"), "")
		return
	}
	loginInf := service.UserSvc{}
	result, err := loginInf.GetCurUser(token)
	if err != nil {
		SendJsonResponse(c, err, "")
		return
	}
	SendJsonResponse(c, resp.OK, result)
}

func GetCaptcha(c *gin.Context) {
	widthS := c.Query("width")
	heightS := c.Query("height")
	var width, height int
	if widthS == "" {
		width = conf.WIDTH
	} else {
		width, _ = strconv.Atoi(widthS)
	}
	if heightS == "" {
		height = conf.HEIGHT
	} else {
		height, _ = strconv.Atoi(heightS)
	}
	loginInf := service.UserSvc{}
	m, err := loginInf.GetCaptcha(width, height)
	if err != nil {
		SendJsonResponse(c, err, "")
		return
	}
	SendJsonResponse(c, resp.OK, m)
}

func VerifyCaptcha(c *gin.Context) {
	id := c.Query("id")
	value := c.Query("value")
	if id == "" || value == "" {
		SendJsonResponse(c, resp.FAIL.WithMsg("id and value are required"), "")
		return
	}
	loginInf := service.UserSvc{}
	err := loginInf.VerifyCaptcha(id, value)
	if err != nil {
		SendJsonResponse(c, err, "")
		return
	}
	SendJsonResponse(c, resp.OK, "验证码正确")
}

func UserLogout(c *gin.Context) {
	cookie, _ := c.Cookie(conf.CookieName)
	c.SetCookie(conf.CookieName, "0", -1, "/", "", false, true)
	c.SetCookie(conf.CookieNameLastLogin, "0", -1, "/", "", false, true)
	loginInf := service.UserSvc{}
	msg, err := loginInf.UserLogout(cookie)
	if err != nil {
		SendJsonResponse(c, err, "")
		return
	}
	SendJsonResponse(c, resp.OK, msg)
}

// 获取短信验证码
func GetSmsVerifyCode(c *gin.Context) {
	req := request.SendPhoneMsg{}
	err := c.BindJSON(&req)
	if err != nil {
		SendJsonResponse(c, resp.InvalidParam.ErrorDetail(err), nil)
		return
	}
	if !util.CheckMobile(req.Phone) {
		SendJsonResponse(c, errors.New("手机号不合法！"), nil)
		return
	}
	loginInf := service.UserSvc{}
	code, err := loginInf.GetSmsVerifyCode(req.Phone)
	if err != nil {
		SendJsonResponse(c, err, nil)
		return
	}
	SendJsonResponse(c, resp.OK, code)
}

func UserPhoneLogin(c *gin.Context) {
	req := request.SendPhoneMsg{}
	err := c.BindJSON(&req)
	if err != nil {
		SendJsonResponse(c, resp.InvalidParam.ErrorDetail(err), nil)
		return
	}
	if !util.CheckMobile(req.Phone) {
		SendJsonResponse(c, errors.New("手机号不合法！"), nil)
		return
	}
	loginInf := service.UserSvc{}
	userInfo, err := loginInf.GetUserInfo(req.Phone)
	if req.Phone != userInfo.Phone {
		SendJsonResponse(c, resp.DbSelectError.ErrorDetail(errors.New("手机号不正确！")), nil)
		return
	}
	// 获取系统配置信息
	op := service.SystemOptionsSvc{}
	config, err := op.GetSystemOptions()
	if err != nil {
		SendJsonResponse(c, err, nil)
		return
	}

	// TODO STEP2 用户和ip没有加入到访问规则登录不上提醒
	if config.AccessRuleState == 1 {
		ip := util.RemoteIp(c.Request)
		svc := service.AccessRuleSvc{}
		if errCode := svc.CheckIp(ip, userInfo.Id); errCode != nil {
			SendJsonResponse(c, errCode, nil)
			return
		}

	}
	// TODO STEP3 密码强弱提醒
	//  密码强弱提醒
	if config.ForceUpdateState == 1 && config.MinPwdLevel > userInfo.PwdLevel {
		SendJsonResponse(c, resp.FAIL.ErrorDetail(errors.New("密码强度弱，请联系管理员修改密码！")), nil)
		return
	}
	// TODO STEP4 验证码正确性校验
	exist, err := loginInf.VerifyPhoneCode(req.Phone)
	if err != nil {
		SendJsonResponse(c, resp.FAIL.ErrorDetail(err), nil)
		return
	}
	if exist != 1 {
		SendJsonResponse(c, resp.FAIL.ErrorDetail(errors.New("验证码无效")), nil)
		return
	}

	lastLogin, msg, uuidStr, err := loginInf.UserPhoneLogin(req, userInfo)
	if err != nil {
		SendJsonResponse(c, err, nil)
		return
	}
	c.SetCookie(conf.CookieName, uuidStr, 1*60*60*24, "/", "", false, false)
	c.SetCookie(conf.CookieNameLastLogin, lastLogin, 1*60*60*24, "/", "", false, false)
	header.AddUserLoginLog("登入", "phone/login", constant.OpTypeIntMap[constant.Login], c, userInfo)

	SendJsonResponse(c, resp.OK, msg)
}

// 根据手机号获取账户
func PhoneToAccount(c *gin.Context) {
	phone := c.Query("phone")
	if !util.CheckMobile(phone) {
		SendJsonResponse(c, errors.New("手机号不合法！"), nil)
		return
	}
	loginInf := service.UserSvc{}
	data, err := loginInf.PhoneToAccount(phone)
	if err != nil {
		SendJsonResponse(c, err, nil)
		return
	}
	SendJsonResponse(c, resp.OK, data)
}

// 验证码校验
func ForgetPwdCheck(c *gin.Context) {
	req := request.ForgetPwdCheck{}
	err := c.BindJSON(&req)
	if err != nil {
		SendJsonResponse(c, resp.InvalidParam.ErrorDetail(err), nil)
		return
	}
	if err := vd.Validate(req); err != nil {
		SendJsonResponse(c, resp.InvalidParam.ErrorDetail(err), "")
		return
	}
	loginInf := service.UserSvc{}
	err = loginInf.ForgetPwdCheck(req)
	if err != nil {
		SendJsonResponse(c, err, nil)
		return
	}
	SendJsonResponse(c, resp.OK, nil)
}

// 忘记密码
func UpdateAccountPwd(c *gin.Context) {
	req := request.AccountForgetPasswordReq{}
	err := c.BindJSON(&req)
	if err != nil {
		SendJsonResponse(c, resp.InvalidParam.ErrorDetail(err), nil)
		return
	}
	if err := vd.Validate(req); err != nil {
		SendJsonResponse(c, resp.InvalidParam.ErrorDetail(err), "")
		return
	}
	loginInf := service.UserSvc{}
	err = loginInf.UpdateAccountPwd(req)
	if err != nil {
		SendJsonResponse(c, err, nil)
		return
	}
	SendJsonResponse(c, resp.OK, nil)
}
