package controller

import (
	"errors"
	vd "github.com/bytedance/go-tagexpr/validator"
	"github.com/gin-gonic/gin"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/bean/entity"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/pkg/beagle/resp"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/service"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/util"
)

// 获取系统首选项信息
func GetSystemOptions(c *gin.Context) {
	svc := service.SystemOptionsSvc{}
	if systemOptions, err := svc.GetSystemOptions(); err != nil {
		SendJsonResponse(c, err, nil)
	} else {
		SendJsonResponse(c, nil, systemOptions)
	}
}

// 设置授权信息
func SetLicenseOptions(c *gin.Context) {
	var licenseOpts entity.SystemPreferenceConfig
	if err := c.ShouldBindJSON(&licenseOpts); err != nil {
		SendJsonResponse(c, resp.InvalidParam.WithError(err), nil)
		return
	}
	if err := util.ValidateSimple(licenseOpts, "Id"); err != nil {
		SendJsonResponse(c, resp.InvalidParam.WithError(err), nil)
		return
	}
	svc := service.SystemOptionsSvc{}
	if err := svc.SetLicenseOptions(licenseOpts); err != nil {
		SendJsonResponse(c, err, nil)
	} else {
		SendJsonResponse(c, resp.OK, nil)
	}
}

// 设置登陆页信息
func SetLoginPageOptions(c *gin.Context) {
	var licenseOpts entity.SystemPreferenceConfig
	if err := c.ShouldBindJSON(&licenseOpts); err != nil {
		SendJsonResponse(c, resp.InvalidParam.WithError(err), nil)
		return
	}
	if err := util.ValidateSimple(licenseOpts, "Id"); err != nil {
		SendJsonResponse(c, resp.InvalidParam.WithError(err), nil)
		return
	}
	svc := service.SystemOptionsSvc{}
	if err := svc.SetLoginPageOptions(licenseOpts); err != nil {
		SendJsonResponse(c, err, nil)
	} else {
		SendJsonResponse(c, resp.OK, nil)
	}
}

// 设置安全选项
func SetSafeOptions(c *gin.Context) {
	var safeOpts entity.SystemPreferenceConfig
	if err := c.ShouldBindJSON(&safeOpts); err != nil {
		SendJsonResponse(c, resp.InvalidParam.WithError(err), nil)
		return
	}
	if err := util.ValidateSimple(safeOpts, "id", "force_update_state", "min_pwd_level", "session_validity", "pwd_validity", "access_rule_state"); err != nil {
		SendJsonResponse(c, resp.InvalidParam.WithError(err), nil)
		return
	}
	svc := service.SystemOptionsSvc{}
	if err := svc.SetSafeOptions(safeOpts); err != nil {
		SendJsonResponse(c, err, nil)
	} else {
		SendJsonResponse(c, resp.OK, nil)
	}
}

// 设置登录配置
func SetLoginOptions(c *gin.Context) {
	var loginOpts entity.SystemPreferenceConfig
	if err := c.ShouldBindJSON(&loginOpts); err != nil {
		SendJsonResponse(c, resp.InvalidParam.WithError(err), nil)
		return
	}
	if err := util.ValidateSimple(loginOpts, "id", "login_config_state"); err != nil {
		SendJsonResponse(c, resp.InvalidParam.WithError(err), nil)
		return
	}
	if loginOpts.LoginConfigState == 1 {
		if loginOpts.LoginLimitTime == 0 || loginOpts.LoginPwdError == 0 || loginOpts.LoginLockTime == 0 {
			SendJsonResponse(c, resp.InvalidParam.WithError(errors.New("登录受限参数不能为空")), nil)
			return
		}
	}
	svc := service.SystemOptionsSvc{}
	if err := svc.SetLoginOptions(loginOpts); err != nil {
		SendJsonResponse(c, err, nil)
	} else {
		SendJsonResponse(c, resp.OK, nil)
	}
}

// 校验license是否过期
func CheckLicense(c *gin.Context) {
	value, _ := c.Get("user_info")
	user := value.(entity.SystemUserInfo)
	svc := service.SystemOptionsSvc{}
	if data, err := svc.CheckLicense(user); err != nil {
		SendJsonResponse(c, err, nil)
	} else {
		SendJsonResponse(c, nil, data)
	}
}

// 一键还原受限
func ReStartConfig(c *gin.Context) {
	svc := service.SystemOptionsSvc{}
	if data, err := svc.DeleteLockUser(); err != nil {
		SendJsonResponse(c, err, nil)
	} else {
		SendJsonResponse(c, resp.OK, data)
	}
}

func GetLicenseInfo(c *gin.Context) {
	license := entity.LicenseInfo{}
	if err := c.ShouldBindJSON(&license); err != nil {
		SendJsonResponse(c, resp.InvalidParam.WithError(err), nil)
		return
	}
	if err := vd.Validate(license); err != nil {
		SendJsonResponse(c, resp.InvalidParam.WithError(err), "")
		return
	}
	svc := service.SystemOptionsSvc{}
	if data, err := svc.GetLicenseInfo(license); err != nil {
		SendJsonResponse(c, err, nil)
	} else {
		SendJsonResponse(c, resp.OK, data)
	}
}
