package controller

import (
	"encoding/json"
	"errors"
	"fmt"
	"github.com/ghodss/yaml"
	"github.com/gin-gonic/gin"
	"github.com/minio/minio-go/v7"
	"github.com/spf13/cast"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/bean/vo/request"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/common/client"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/common/conf"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/pkg/beagle/resp"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/router/middleware/header"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/service"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/util"
	"io"
	"strings"
)

// AddTaskManage 新增任务
func AddTaskManage(c *gin.Context) {
	var req request.AddTaskManageReq
	if err := c.ShouldBindJSON(&req); err != nil {
		SendJsonResponse(c, resp.InvalidParam.WithError(err), nil)
		return
	}
	//参数校验
	if err := util.ValidateSimple(req, "TaskName,HostGroupId"); err != nil {
		SendJsonResponse(c, resp.InvalidParam.WithError(err), nil)
		return
	}

	taskManageSvc := service.TaskManageSvc{User: header.GetUser(c)}
	id, err := taskManageSvc.AddTaskManage(req)
	if err != nil {
		SendJsonResponse(c, err, nil)
		return
	}
	SendJsonResponse(c, resp.OK, id)
}

// EditTaskManage 编辑任务
func EditTaskManage(c *gin.Context) {
	var req request.EditTaskManageReq
	if err := c.ShouldBindJSON(&req); err != nil {
		SendJsonResponse(c, resp.InvalidParam.WithError(err), nil)
		return
	}
	//参数校验
	if err := util.ValidateSimple(req, "Id,HostGroupId"); err != nil {
		SendJsonResponse(c, resp.InvalidParam.WithError(err), nil)
		return
	}

	taskManageSvc := service.TaskManageSvc{User: header.GetUser(c)}
	id, err := taskManageSvc.EditTaskManage(req)
	if err != nil {
		SendJsonResponse(c, err, nil)
		return
	}
	SendJsonResponse(c, resp.OK, id)
}

// DelTaskManage 删除任务
func DelTaskManage(c *gin.Context) {
	var req request.DelTaskManageReq
	if err := c.ShouldBindJSON(&req); err != nil {
		SendJsonResponse(c, resp.InvalidParam.WithError(err), nil)
		return
	}
	//参数校验
	if err := util.ValidateSimple(req, "Id"); err != nil {
		SendJsonResponse(c, resp.InvalidParam.WithError(err), nil)
		return
	}

	taskManageSvc := service.TaskManageSvc{}
	err := taskManageSvc.DelTaskManage(req)
	if err != nil {
		SendJsonResponse(c, err, nil)
		return
	}
	SendJsonResponse(c, resp.OK, nil)
}

// DetailsTaskManage 任务详情
func DetailsTaskManage(c *gin.Context) {
	var (
		err error
		id  string
	)

	if id = c.Query("id"); id == "" {
		id = c.Param("id")
	}
	if id == "" {
		SendJsonResponse(c, resp.InvalidParam.WithError(errors.New("id为空")), nil)
		return
	}

	taskManageSvc := service.TaskManageSvc{}
	data, err := taskManageSvc.DetailsTaskManage(cast.ToInt(id))
	if err != nil {
		SendJsonResponse(c, err, nil)
		return
	}
	SendJsonResponse(c, resp.OK, data)
}

// ListTaskManage 任务列表
func ListTaskManage(c *gin.Context) {
	var req request.ListTaskManageReq
	if err := c.ShouldBind(&req); err != nil {
		SendJsonResponse(c, resp.InvalidParam.WithError(err), nil)
		return
	}

	taskManageSvc := service.TaskManageSvc{}
	total, list, err := taskManageSvc.ListTaskManage(req)
	if err != nil {
		SendJsonPageResponse(c, err, nil, 0)
		return
	}
	SendJsonPageResponse(c, resp.OK, list, total)
}

func ExecScript(c *gin.Context) {
	var (
		req request.ExecScriptReq
		err error
	)
	if err = c.ShouldBind(&req); err != nil {
		SendJsonResponse(c, resp.InvalidParam.WithError(err), nil)
		return
	}

	taskManageSvc := service.TaskManageSvc{User: header.GetUser(c)}
	taskManage, err := taskManageSvc.GetTaskManage(req.TaskId)
	if err != nil {
		SendJsonResponse(c, err, nil)
		return
	}

	if taskManage.YamlDesc == "" {
		//解析minio文件
		ymlUrl := strings.Split(taskManage.YamlUrl, "/")
		minioClient, err2 := client.GetMinioConnect()
		if err2 != nil {
			SendJsonResponse(c, resp.InvalidParam.WithError(err2), nil)
			return
		}
		object, err2 := minioClient.GetObject(c, conf.Options.MinioBucket, ymlUrl[len(ymlUrl)-1], minio.GetObjectOptions{})
		if err2 != nil {
			SendJsonResponse(c, resp.GetFileStreamError.WithError(err2), nil)
			return
		}
		obj, err2 := io.ReadAll(object)
		if err2 != nil {
			SendJsonResponse(c, resp.ReadFileError.WithError(err2), nil)
			return
		}
		taskManage.YamlDesc = string(obj)
	}

	//写入额外yml变量
	if req.Type == 1 {
		var value map[string]interface{}
		j2, err3 := yaml.YAMLToJSON([]byte(req.Value))
		if err3 != nil {
			err = resp.YamlAnalysisError.WithError(err3)
			return
		}
		err3 = json.Unmarshal(j2, &value)
		if err3 != nil {
			err = resp.InvalidParam.WithError(errors.New("变量配置错误"))
			return
		}

		j, err3 := json.Marshal(value)
		if err3 != nil {
			err = resp.MarshalError.WithError(err3)
			return
		}
		req.Value = fmt.Sprintf("%s", j)
	}

	id, err := taskManageSvc.ExecScript(req, taskManage.YamlDesc)
	if err != nil {
		SendJsonResponse(c, err, nil)
		return
	}
	SendJsonResponse(c, nil, id)
}
