package main

import (
	"fmt"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/service/cron"
	"net/http"
	"os"
	"time"

	"github.com/gin-gonic/gin"
	"github.com/spf13/pflag"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/common/client"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/common/conf"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/pkg/beagle/log"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/router"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/util"
	"go.uber.org/zap"
	"go.uber.org/zap/zapcore"
)

// run args
var (
	port   = pflag.Int("port", 80, "server port")
	prefix = pflag.String("prefix", "/v1/api", "url prefix")
)

// main start
func main() {
	initTimeZone()
	// init ansible hosts
	initAnsibleHosts()
	initTempDirPrefix()
	pflag.Parse() // init start args
	initConfig()

	initAnsibleSSH()
	// init log config
	cfg := initLogConfig()
	conf.Logger = log.NewLogger(cfg, conf.LoggerLevel)
	// init validator
	util.RegValid()
	// db client
	go client.GetDbClient()
	// redis client
	go client.GetRedisClient()

	//启动定时任务
	cron.StartCron()

	// server start...
	conf.Logger.Error("server start err", zap.Error(newServer().ListenAndServe()))
}

// init commonutil config value
func initConfig() {
	conf.LoggerLevel = util.SetEnvStr("LOG_LEVEL", "info")
	conf.RunMode = util.SetEnvStr("GIN_MODE", "debug") // project run mode, available parameters: debug、release
	conf.Options = &conf.Config{
		Prefix:                   *prefix,
		DbURL:                    util.SetEnvStr("DB_URL", "host=localhost port=9010 user=postgres password=passwd123 dbname=postgres sslmode=disable"),
		DbDriver:                 util.SetEnvStr("DB_DRIVER", "postgres"),
		RedisURL:                 util.SetEnvStr("REDIS_URL", "localhost:7001"),
		RedisDB:                  0,
		RedisTag:                 "bg",
		LogDirPrefix:             util.SetEnvStr("LOG_DIR_PREFIX", "/app/log"),                         // 日志目录
		LogDirName:               util.SetEnvStr("LOG_NAME", "syslog"),                                 // 日志名称
		LogSaveDays:              util.SetEnvInt("LOG_SAVE_DAYS", 7),                                   // 日志最大存储天数
		LogMode:                  util.SetEnvInt("LOG_MODE", 1),                                        // 1.标准打印 2.输出文件
		ArgBool:                  util.SetEnvBool("ARG_BOOL", false),                                   // 示例参数
		ArgInt:                   util.SetEnvInt("ARG_INT", 10),                                        // 示例参数
		MinioServer:              util.SetEnvStr("MINIO_SERVER", "https://cache.wodcloud.com"),         // Minio 服务地址
		MinioAccessKey:           util.SetEnvStr("MINIO_ACCESS_KEY", "beagleadmin"),                    // Minio Access Key
		MinioSecretKey:           util.SetEnvStr("MINIO_SECRET_KEY", "H76cPmwvH7vJ"),                   // Minio Secret
		MinioBucket:              util.SetEnvStr("MINIO_BUCKET", "so-operation"),                       // Minio Bucket
		PrometheusHost:           util.SetEnvStr("PROMETHEUS_HOST", "https://prometheus.wodcloud.com"), // Prometheus Host
		AccessRuleModeKey:        "accessRuleMode",
		LocationUrl:              util.SetEnvStr("LOCATION_URL", "https://apis.map.qq.com/ws/location/v1/ip"),
		LocationKey:              util.SetEnvStr("LOCATION_KEY", "QKFBZ-PGGWJ-VZQFF-FHPA7-QWT5H-YHF4T"),
		PrivateKeySSH:            util.SetEnvStr("PRIVATE_KEY_SSH", "-----BEGIN OPENSSH PRIVATE KEY-----\nb3BlbnNzaC1rZXktdjEAAAAABG5vbmUAAAAEbm9uZQAAAAAAAAABAAAAlwAAAAdzc2gtcn\nNhAAAAAwEAAQAAAIEAsOFk9OUB8wg9fd+PDHyX8nEtTSPSZY+tjxq2da1Pf5FkIn+U1da6\nh2eqowF9lnyvlt7uEledTIWQZDGWToGYCZnRommSZEpo/vII+l1P28bJVHfgWFCqmxNfIB\nZFQ4KrOp9rXKidmrd8flhK/NTLJNqryrhhIiDs3CTyAliscIsAAAIQTuM2gU7jNoEAAAAH\nc3NoLXJzYQAAAIEAsOFk9OUB8wg9fd+PDHyX8nEtTSPSZY+tjxq2da1Pf5FkIn+U1da6h2\neqowF9lnyvlt7uEledTIWQZDGWToGYCZnRommSZEpo/vII+l1P28bJVHfgWFCqmxNfIBZF\nQ4KrOp9rXKidmrd8flhK/NTLJNqryrhhIiDs3CTyAliscIsAAAADAQABAAAAgDjcfGPtqq\n7CG2J3l7jf5MjfcTy3I0/a3GSApd82k7PivVoJwYLswJH+1XAJbqIN+zR4/fePitWqqjxL\nZJJgPstuXpBZuJDvGwMqfl7wHRL2Qx34sRG02hG5e3uIfMxe5lHcPba0qsVQt+vOhu9MUb\nsYF/mfuQJKt/Oi8nA1BbrBAAAAQFQPrap7AtYWEoCIY7gtpFMW51iDTAv5GN99DsKNuBby\nwQX2S0Wg/da75m/emJn/2IbmaKApvrx8LbenpyywfBkAAABBAN6xiYQ2j7eRjLV4h4Hbie\nVwlPYP4otKHdF5meObr+2ifYiMktdv/44V1XWKhgavjGFNWx2sHgj7byb51e/bi3MAAABB\nAMtVxa55G0wS9Yw1WK2F4JdYZ65ZAnUuo2rbA2dMDQxsOQxgel5Ox2XmC7e0GKrO9BJKPo\nR2fHEOdm9KOmoB8IkAAAAWY2hlbnppbG9uZ0BleGFtcGxlLmNvbQECAwQF\n-----END OPENSSH PRIVATE KEY-----\n"),
		PublicKeySSH:             util.SetEnvStr("PUBLIC_KEY_SSH", "ssh-rsa AAAAB3NzaC1yc2EAAAADAQABAAAAgQCw4WT05QHzCD19348MfJfycS1NI9Jlj62PGrZ1rU9/kWQif5TV1rqHZ6qjAX2WfK+W3u4SV51MhZBkMZZOgZgJmdGiaZJkSmj+8gj6XU/bxslUd+BYUKqbE18gFkVDgqs6n2tcqJ2at3x+WEr81Msk2qvKuGEiIOzcJPICWKxwiw== chenzilong@example.com\n"),
		OrderDingTalkAccessToken: util.SetEnvStr("ORDER_DING_TALK_ACCESS_TOKEN", "203fe1644b446bba0a34e6e622c523d39ee9916fdad94b9c64224449f659e20b"),
		OrderDingTalkSecret:      util.SetEnvStr("ORDER_DING_TALK_SECRET", "SECa73d8372e336451c9daf29a99f750ee1bdd170c1dab910eab9cd06d729a831b7"),
		SmsAccessKeyId:           util.SetEnvStr("SMS_ACCESS_KEY", "LTAI4GBcVubRjzX7ABPcHnhB"),          // 短信key
		SmsAccessSecret:          util.SetEnvStr("SMS_ACCESS_SECRET", "dYE2dtABFOqYtK1ijcrits0yedHkw7"), // 短信secret
		SmsTemplateLogin:         util.SetEnvStr("SMS_TEMPLATE_LOGIN", "SMS_212925130"),                 // 短信验证码模板
		SmsSignName:              util.SetEnvStr("SMS_SIGN_NAME", "比格数据"),                               // 签名
	}
}

// init log config
func initLogConfig() zapcore.EncoderConfig {
	return zapcore.EncoderConfig{
		LevelKey:       "level",
		TimeKey:        "time",
		MessageKey:     "msg",
		NameKey:        "logger",
		CallerKey:      "caller",
		StacktraceKey:  "stacktrace",
		LineEnding:     zapcore.DefaultLineEnding,
		EncodeLevel:    zapcore.CapitalLevelEncoder,
		EncodeTime:     zapcore.TimeEncoderOfLayout("2006-01-02 15:04:05"),
		EncodeDuration: zapcore.SecondsDurationEncoder,
		EncodeCaller:   zapcore.ShortCallerEncoder,
	}
}

func newServer() *http.Server {
	// set run mod
	gin.SetMode(conf.RunMode)
	// load gin router
	r := gin.New()
	router.Load(r, log.GinLogger(), log.GinRecovery(true))
	conf.Logger.Info("server is starting...", zap.Int("port", *port))
	return &http.Server{
		Addr:           fmt.Sprintf(":%d", *port),
		Handler:        r,
		ReadTimeout:    30 * time.Second,
		WriteTimeout:   30 * time.Second,
		MaxHeaderBytes: 1 << 20,
	}
}

func initTimeZone() {
	//var cstZone, _ = time.LoadLocation("Asia/Shanghai")
	var cstZone = time.FixedZone("CST", 8*3600) // 固定东八区(CST: China Standard Time)
	time.Local = cstZone
}

func initAnsibleHosts() {
	err := os.Mkdir("/etc/ansible", os.ModePerm)
	if err != nil {
		fmt.Println("创建目录失败!")
	}
	f, err := os.Create("/etc/ansible/hosts")
	defer f.Close()
	if err != nil {
		fmt.Println(err.Error())
	}
	//else {
	//	_, err := f.Write([]byte("[web]\n"))
	//	if err != nil {
	//		fmt.Println(err.Error())
	//	}
	//}
}

func initTempDirPrefix() {
	err := os.Mkdir("/app/xlsx/", os.ModePerm)
	if err != nil {
		fmt.Println("创建目录失败!")
	}
}

func initAnsibleSSH() {
	err := os.Mkdir("/root/.ssh", os.ModePerm)
	if err != nil {
		fmt.Println("创建目录失败!")
	}
	f, err := os.Create("/root/.ssh/id_rsa")
	defer f.Close()
	if err != nil {
		fmt.Println(err.Error())
	} else {
		_, err := f.Write([]byte(conf.Options.PrivateKeySSH))
		if err != nil {
			fmt.Println(err.Error())
		}
	}
	f2, err := os.Create("/root/.ssh/id_rsa.pub")
	defer f2.Close()
	if err != nil {
		fmt.Println(err.Error())
	} else {
		_, err := f.Write([]byte(conf.Options.PublicKeySSH))
		if err != nil {
			fmt.Println(err.Error())
		}
	}
}
