package header

import (
	"bytes"
	"encoding/json"
	"errors"
	"fmt"
	"github.com/gin-gonic/gin"
	"github.com/spf13/cast"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/bean/entity"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/bean/vo/response"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/common/conf"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/pkg/beagle/constant"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/pkg/beagle/resp"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/service"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/util"
	"go.uber.org/zap"
	"io/ioutil"
	"strings"
	"time"
)

type responseBodyWriter struct {
	gin.ResponseWriter
	body *bytes.Buffer
}

func (r responseBodyWriter) Write(b []byte) (int, error) {
	r.body.Write(b)
	return r.ResponseWriter.Write(b)
}

func CopyResponseBody(c *gin.Context) {
	w := &responseBodyWriter{body: &bytes.Buffer{}, ResponseWriter: c.Writer}
	c.Writer = w
	c.Next()
}

// 添加日志
func AddLogMiddleware(sysModule string, operateMethod string, operatorType int) func(ctx *gin.Context) {
	return func(c *gin.Context) {
		reqBody, err := c.GetRawData()
		if err != nil {
			conf.Logger.Error("SaveUserOpLog err", zap.Error(err))
		}
		c.Request.Body = ioutil.NopCloser(bytes.NewBuffer(reqBody))

		c.Next()
		go func() {
			var (
				ip         = util.RemoteIp(c.Request)
				url        = c.Request.RequestURI
				opStats    = int8(1)
				resData    = resp.Resp{}
				resBody    = make([]byte, 0)
				httpMethod = c.Request.Method
				userInfo   = entity.SystemUserInfo{}
				appId      = c.Query("app_id")
			)
			if c.Writer.Status() == 404 {
				return
			}
			if c.Writer.Status() >= 400 {
				opStats = 2
			}
			if val, ok := c.Writer.(*responseBodyWriter); ok {
				resBody = val.body.Bytes()
				// 解析body
				err := json.Unmarshal(resBody, &resData)
				if err != nil {
					resBody = nil
					conf.Logger.Error("Unmarshal.err", zap.Error(err))
				} else if resData.Code != 200 {
					opStats = 2
				}
			}
			token, _ := c.Cookie(conf.CookieName)
			conf.Logger.Info("当前token信息为", zap.String("bgToken", token))
			if token == "" {
				conf.Logger.Error("登录超时", zap.Error(errors.New("登录超时")))
			}
			loginInf := service.UserSvc{}
			userInfo, err = loginInf.GetCurUser(token)
			if err != nil || userInfo.Id == 0 {
				conf.Logger.Error("获取用户信息失败")
			}
			log := service.LogSvc{}
			logData := entity.SystemUserBehavior{
				UserId:         userInfo.Id,
				OrganizationId: userInfo.OrganizationId,
				SystemModule:   sysModule,
				OperateType:    constant.OpTypeMap[operatorType],
				ReqMethod:      cast.ToInt(constant.HttpMethodMap[httpMethod]),
				OperateStatus:  cast.ToInt(opStats),
				OperateIp:      ip,
				OperateAddr:    getAddressByIp(ip),
				CreatedTime:    time.Now(),
				OperateMethod:  operateMethod,
				ResFields:      util.Bytes2Str(resBody),
				ReqParam:       util.Bytes2Str(reqBody),
				ReqUrl:         url,
				AppId:          appId,
			}
			//是否切换ES查询

			log.AddBehaviorLog(logData)

		}()
	}
}

// 添加登录日志
func AddUserLoginLog(sysModule string, operateMethod string, operatorType int, c *gin.Context, userInfo entity.SystemUserInfo) {
	reqBody, err := c.GetRawData()
	if err != nil {
		conf.Logger.Error("SaveUserOpLog err", zap.Error(err))
	}
	c.Request.Body = ioutil.NopCloser(bytes.NewBuffer(reqBody))
	go func() {
		var (
			ip         = util.RemoteIp(c.Request)
			url        = c.Request.RequestURI
			opStats    = int8(1)
			resData    = resp.Resp{}
			resBody    = make([]byte, 0)
			httpMethod = c.Request.Method
			appId      = c.Query("app_id")
		)
		if c.Writer.Status() == 404 {
			return
		}
		if c.Writer.Status() >= 400 {
			opStats = 2
		}
		if val, ok := c.Writer.(*responseBodyWriter); ok {
			resBody = val.body.Bytes()
			// 解析body
			err := json.Unmarshal(resBody, &resData)
			if err != nil {
				resBody = nil
				conf.Logger.Error("Unmarshal.err", zap.Error(err))
			} else if resData.Code != 200 {
				opStats = 2
			}
		}
		log := service.LogSvc{}
		logData := entity.SystemUserBehavior{
			UserId:         userInfo.Id,
			OrganizationId: userInfo.OrganizationId,
			SystemModule:   sysModule,
			OperateType:    constant.OpTypeMap[operatorType],
			ReqMethod:      cast.ToInt(constant.HttpMethodMap[httpMethod]),
			OperateStatus:  cast.ToInt(opStats),
			OperateIp:      ip,
			OperateAddr:    getAddressByIp(ip),
			CreatedTime:    time.Now(),
			OperateMethod:  operateMethod,
			ResFields:      util.Bytes2Str(resBody),
			ReqParam:       util.Bytes2Str(reqBody),
			ReqUrl:         url,
			AppId:          appId,
			Phone:          userInfo.Phone,
		}

		log.AddBehaviorLog(logData)

	}()
}

// 根据ip地址获取登录地点
func getAddressByIp(ip string) string {
	if ip == "" {
		return "未知"
	}
	if strings.HasPrefix(ip, "127") || strings.HasPrefix(ip, "10") || strings.HasPrefix(ip, "172") || strings.HasPrefix(ip, "192") {
		return "局域网"
	}
	requestUrl := fmt.Sprintf("%s?ip=%s&key=%s", conf.Options.LocationUrl, ip, conf.Options.LocationKey)
	resData, err := util.HttpSend("GET", requestUrl, "", nil)
	if err != nil {
		fmt.Println("request location url err:", err)
		return "未知"
	}
	body, _ := ioutil.ReadAll(resData.Body)
	var result response.LocationResult
	if err = json.Unmarshal(body, &result); err != nil {
		fmt.Println("parse location resp err:", err)
		return "未知"
	}
	var address string
	if result.Result.AdInfo.Province == result.Result.AdInfo.City {
		address = result.Result.AdInfo.Province
	} else {
		address = result.Result.AdInfo.Province + result.Result.AdInfo.City
	}
	if address == "" {
		return "未知"
	} else {
		return address
	}
}
