package service

import (
	"errors"
	"fmt"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/bean/entity"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/bean/vo/request"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/bean/vo/response"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/common/client"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/common/conf"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/pkg/beagle/resp"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/util"
	"xorm.io/xorm"

	"go.uber.org/zap"
)

type Organization struct {
	User entity.SystemUserInfo
}

// 新增目录或组织
func (o *Organization) AddOrg(input request.OrgInput) (interface{}, error) {
	db, err := client.GetDbClient()
	if err != nil {
		return nil, resp.DbConnectError.WithError(err)
	}

	//数据校验
	if input.DataType == 1 {
		has, err := db.Table("system_organization").Select("id").Where("organization_code = ? and is_deleted = 0", input.OrganizationCode).Exist(&entity.SystemOrganization{})
		if err != nil {
			conf.Logger.Error("组织机构代码获取组织信息失败", zap.Error(err))
			return nil, resp.DbSelectError.WithError(err)
		}
		if has {
			return nil, resp.DbInsertError.WithError(errors.New("组织机构代码已存在"))
		}
	}

	orgModel := entity.SystemOrganization{
		Id:               0,
		OrganizationId:   util.GetUUID(),
		Name:             input.Name,
		POrganizationId:  input.POrganizationId,
		Description:      input.Description,
		OrganizationCode: input.OrganizationCode,

		DataType:  input.DataType,
		CreatedBy: o.User.Id,
	}
	var samelevelOrg, parentOrg entity.SystemOrganization
	if orgModel.POrganizationId != "" {
		has, err := db.Table("system_organization").Where("p_organization_id = ? and is_deleted = 0", orgModel.POrganizationId).Limit(1, 0).OrderBy("sort desc").Get(&samelevelOrg)
		if err != nil {
			conf.Logger.Error("获取同级组织信息失败", zap.Error(err))
			return nil, resp.DbConnectError.WithError(err)
		}

		if !has {
			_, err := db.Table("system_organization").Where("organization_id = ? and is_deleted = 0", orgModel.POrganizationId).Limit(1, 0).OrderBy("sort desc").Get(&parentOrg)
			if err != nil {
				conf.Logger.Error("获取父级组织信息失败", zap.Error(err))
				return nil, resp.DbConnectError.WithError(err)
			}
			orgModel.Level = parentOrg.Level + 1
			orgModel.Sort = 1
		} else {
			orgModel.Level = samelevelOrg.Level
			orgModel.Sort = samelevelOrg.Sort + 1
		}
	} else {
		//查询跟目录sort
		has, err := db.Table("system_organization").Where("is_deleted = 0 and level = 1").Limit(1, 0).OrderBy("sort desc").Get(&parentOrg)
		if err != nil {
			conf.Logger.Error("获取同级组织信息失败", zap.Error(err))
			return nil, resp.DbConnectError.WithError(err)
		}
		if has {
			orgModel.Sort = parentOrg.Sort + 1
		} else {
			orgModel.Sort = 1
		}
		orgModel.Level = 1
	}
	if _, err := db.Table("system_organization").Omit("id").Insert(&orgModel); err != nil {
		conf.Logger.Error("插入 system_organization", zap.String("error", err.Error()))
		return nil, resp.DbInsertError.WithError(err)
	}
	result := map[string]interface{}{"id": orgModel.Id, "organization_id": orgModel.OrganizationId}
	return result, nil
}

// 更新组织或目录
func (o *Organization) UpdateOrg(id int64, input request.OrgInput) error {
	db, err := client.GetDbClient()
	if err != nil {
		return resp.DbConnectError.WithError(err)
	}

	var org entity.SystemOrganization
	has, err := db.Table("system_organization").ID(id).Where("is_deleted = 0").Get(&org)
	if err != nil {
		conf.Logger.Error("获取组织信息失败", zap.Error(err))
		return resp.DbSelectError.WithError(err)
	}
	if !has {
		if input.DataType == 0 {
			return resp.FAIL.WithMsg("该目录不存在")
		}
		if input.DataType == 1 {
			return resp.FAIL.WithMsg("该组织不存在")
		}
	}

	//数据校验
	if input.DataType == 1 && org.OrganizationCode != input.OrganizationCode {
		has, err := db.Table("system_organization").Select("id").Where("organization_code = ? and is_deleted = 0", input.OrganizationCode).Exist(&entity.SystemOrganization{})
		if err != nil {
			conf.Logger.Error("组织机构代码获取组织信息失败", zap.Error(err))
			return resp.DbSelectError.WithError(err)
		}
		if has {
			return resp.DbUpdateError.WithError(errors.New("组织机构代码已存在"))
		}
	}

	if _, err := db.Table("system_organization").Cols("name, description, organization_code,  updated_time, updated_by").
		Where("id = ?", id).Update(&entity.SystemOrganization{
		Name:             input.Name,
		Description:      input.Description,
		OrganizationCode: input.OrganizationCode,
		UpdatedBy:        o.User.Id,
	}); err != nil {
		conf.Logger.Error("更新组织信息失败", zap.Error(err))
		return resp.DbUpdateError.WithError(err)
	}
	return nil
}

// 删除组织或者目录
func (o *Organization) DeleteOrg(id int64) error {
	db, err := client.GetDbClient()
	if err != nil {
		return resp.DbConnectError.WithError(err)
	}
	_, err = db.Transaction(func(session *xorm.Session) (interface{}, error) {
		var org entity.SystemOrganization
		basehas, err := session.Table("system_organization").ID(id).Where("is_deleted = 0").Get(&org)
		if err != nil {
			conf.Logger.Error("查询基础组织信息失败", zap.Error(err))
			return nil, resp.DbSelectError.WithError(err)
		}

		//目录 目录没有下级可删除
		if org.DataType == 0 {
			if !basehas {
				return nil, resp.FAIL.WithMsg("该目录不存在")
			}

			//验证目录是否有下级
			has, err := session.Table("system_organization").Select("id").Where("is_deleted = 0 and p_organization_id = ?", org.OrganizationId).Exist(&entity.SystemOrganization{})
			if err != nil {
				conf.Logger.Error("验证目录是否有下级查询基础组织信息失败", zap.Error(err))
				return nil, resp.DbSelectError.WithError(err)
			}
			if has {
				return nil, resp.FAIL.WithMsg("该目录下有数据，不可删除！")
			}
		}

		//组织 组织没有组织管理员且没有绑定业务系统
		if org.DataType == 1 {
			if !basehas {
				return nil, resp.FAIL.WithMsg("该组织不存在")
			}

			//验证组织下是否有组织管理员且是否绑定业务系统
			has, err := session.Table("system_user").Select("id").Where("is_deleted = 0 and organization_id = ?", org.OrganizationId).Exist(&entity.SystemUser{})
			if err != nil {
				conf.Logger.Error("验证组织下是否有组织管理员查询基础组织信息失败", zap.Error(err))
				return nil, resp.DbSelectError.WithError(err)
			}
			if has {
				return nil, resp.FAIL.WithMsg("该组织下有业务数据，不可删除！")
			}
		}

		//删除
		_, err = session.Table("system_organization").Cols("is_deleted, updated_time, updated_by").ID(id).Update(&entity.SystemOrganization{IsDeleted: 1, UpdatedBy: o.User.Id})
		return nil, err
	})
	return err
}

// 组织排序
func (o *Organization) SortOrg(input []request.OrgSortInput) error {
	db, err := client.GetDbClient()
	if err != nil {
		return resp.DbConnectError.WithError(err)
	}
	_, err = db.Transaction(func(s *xorm.Session) (interface{}, error) {
		for _, v := range input {
			_, err := s.Table("system_organization").Cols("sort, updated_time, updated_by").Where("id = ? and is_deleted = 0", v.Id).Update(&entity.SystemOrganization{Sort: v.Sort, UpdatedBy: o.User.Id})
			if err != nil {
				conf.Logger.Error("排序失败", zap.Error(err))
				return nil, resp.DbUpdateError.WithError(err)
			}
		}
		return nil, nil
	})
	return err
}

// 查询组织树
func (o *Organization) GetOrgTree() (interface{}, error) {
	db, err := client.GetDbClient()
	if err != nil {
		return nil, resp.DbConnectError.WithError(err)
	}
	var orgs []response.SystemOrganizationTree
	if err := db.Table("system_organization").Where("is_deleted = 0").OrderBy("level").OrderBy("sort").Find(&orgs); err != nil {
		conf.Logger.Error("查询组织树失败", zap.Error(err))
		return nil, resp.DbSelectError.WithError(err)
	}

	var result []response.SystemOrganizationTree
	// 组装树形
	for i, v := range orgs {
		if v.Level == 1 {
			changeTree(&orgs[i], orgs)
			result = append(result, orgs[i])
		}
	}
	return result, nil
}

func changeTree(parent *response.SystemOrganizationTree, all []response.SystemOrganizationTree) {
	for i, v := range all {
		if v.POrganizationId == parent.OrganizationId {
			changeTree(&all[i], all)
			parent.Child = append(parent.Child, all[i])
		}
	}
}

// 查询组织详情
func (o *Organization) OrgDetail(input request.QueryOrgDetailInput) (interface{}, error) {
	db, err := client.GetDbClient()
	if err != nil {
		return nil, resp.DbConnectError.WithError(err)
	}

	var orgInfo response.OrgDetail
	if _, err = db.Table("system_organization").Alias("so").
		Select("so.organization_code, so.name, so.created_time,  t3.count as platform_users_number,  so.description, (case when so.name <> '平台用户组织' then '政务机构' else '平台用户组织' end) as organization_type").
		Join("LEFT", "(SELECT count(id),su.organization_id from system_user as su where  su.is_deleted = 0 and su.state = 1 GROUP BY su.organization_id) t3", "t3.organization_id = so.organization_id").
		Where("so.organization_id = ? and is_deleted = 0", input.OrganizationId).Get(&orgInfo); err != nil {
		conf.Logger.Error("查询组织基本信息失败", zap.Error(err))
		return nil, resp.DbSelectError.WithError(err)
	}

	//查询用户 区分组织管理员和平台用户
	var orgAdminUsers []response.OrgAdminUser
	querySystemUser := db.Table("system_user").Alias("su").Select("su.id,su.name, su.system_account,  su.created_time, su.state, su.phone").
		Where("organization_id = ? and su.is_deleted = 0", input.OrganizationId)

	//DataType 2平台用户组织
	//if input.DataType == "2" {
	//	querySystemUser.Where("su.is_admin = 3")
	//} else {
	//	//组织管理员用户
	//	querySystemUser.Where("su.is_admin = 2")
	//}

	if input.KeyWord != "" {
		querySystemUser.Where("su.system_account like ? or su.phone  like ?", fmt.Sprintf("%%%s%%", input.KeyWord), fmt.Sprintf("%%%s%%", input.KeyWord))
	}
	if input.State != "" {
		querySystemUser.Where("su.state = ?", input.State)
	}
	count, err := querySystemUser.Limit(input.Limit, (input.Page-1)*input.Limit).Asc("su.created_time").FindAndCount(&orgAdminUsers)
	if err != nil {
		conf.Logger.Error("查询用户失败", zap.Error(err))
		return nil, resp.DbSelectError.WithError(err)
	}

	//查询用户角色
	var ids []int
	for _, v := range orgAdminUsers {
		ids = append(ids, v.Id)
	}
	var systemUserRoles []response.SystemUserRole
	if err := db.Table("system_user_role").Alias("sur").Select("sur.user_id, sr.role_name").
		Join("left", "system_role sr", "sr.role_id = sur.role_id").In("sur.user_id", ids).Find(&systemUserRoles); err != nil {
		conf.Logger.Error("查询用户角色信息失败", zap.Error(err))
		return nil, resp.DbSelectError.ErrorDetail(err)
	}
	for i, v := range orgAdminUsers {
		orgAdminUsers[i].OrgName = orgInfo.Name
		for _, s := range systemUserRoles {
			if v.Id == s.UserId {
				orgAdminUsers[i].SystemRole = append(orgAdminUsers[i].SystemRole, s.RoleName)
			}
		}
	}

	result := map[string]interface{}{
		"org_info":  orgInfo,
		"org_users": map[string]interface{}{"data": orgAdminUsers, "total": count},
	}
	//if orgInfo.OrganizationCode == "" {
	//	result = nil
	//}
	return result, nil
}
