package service

import (
	"errors"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/bean/entity"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/bean/vo/response"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/common/client"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/common/conf"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/pkg/beagle/license"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/pkg/beagle/resp"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/util"

	"time"

	"go.uber.org/zap"
)

type SystemOptionsSvc struct {
}

// 获取系统首选项
func (so *SystemOptionsSvc) GetSystemOptions() (config *entity.SystemPreferenceConfig, err error) {
	db, err := client.GetDbClient()
	if err != nil {
		return nil, resp.DbConnectError.WithError(err)
	}
	var systemOpts entity.SystemPreferenceConfig
	if _, err := db.Table("system_preference_config").Get(&systemOpts); err != nil {
		conf.Logger.Error("获取系统首选项配置失败", zap.Error(err))
		return nil, resp.DbSelectError.WithError(err)
	}
	return &systemOpts, nil
}

func (so *SystemOptionsSvc) SetLicenseOptions(licenseOpts entity.SystemPreferenceConfig) error {
	if err := util.ValidateSimple(licenseOpts, "Id"); err != nil {
		return resp.InvalidParam.WithError(err)
	}
	// 验证license信息
	if licenseInfo, err := license.DecodeLicense(licenseOpts.License); err != nil {
		conf.Logger.Error("license err", zap.Error(err))
		return resp.InvalidParam.WithError(err)
	} else {
		conf.Logger.Info("license解析信息:", zap.Any("license", licenseInfo))
		if licenseInfo.CustomerName != licenseOpts.SystemUser {
			return resp.InvalidParam.WithError(err)
		}
		if len(licenseInfo.DeadDate) != 11 {
			deadDate, _ := time.Parse("2006-01-02", licenseInfo.DeadDate)
			// It is shorthand for t.Sub(time.Now()).
			if time.Until(deadDate) <= 0 {
				return resp.InvalidParam.WithError(errors.New("license 已过期"))
			}
			licenseOpts.LicenseDeadDate = licenseInfo.DeadDate
		}
	}
	db, err := client.GetDbClient()
	if err != nil {
		return resp.DbConnectError.ErrorDetail(err)
	}
	if _, err := db.Table("system_preference_config").Where("id = ?", licenseOpts.Id).Cols("system_version", "system_user", "license", "license_dead_date", "license_inform_day").Update(licenseOpts); err != nil {
		conf.Logger.Error("dberr", zap.Error(err))
		return resp.DbUpdateError.ErrorDetail(err)
	}
	return nil
}

// 设置登录页
func (so *SystemOptionsSvc) SetLoginPageOptions(safeOpts entity.SystemPreferenceConfig) error {

	// 获取数据库连接
	db, err := client.GetDbClient()
	if err != nil {
		return resp.DbConnectError.ErrorDetail(err)
	}
	_, err = db.Table("system_preference_config").Where("id = ?", safeOpts.Id).
		Cols("logo", "bg_image", "bottom_title", "icp", "icp_url", "police", "police_url", "legal_notice", "privacy_policy").
		Update(safeOpts)
	if err != nil {
		conf.Logger.Error("设置登录页信息失败!", zap.Error(err))
		return resp.DbUpdateError.ErrorDetail(err)
	}

	return nil
}

// 设置安全信息
func (so *SystemOptionsSvc) SetSafeOptions(safeOpts entity.SystemPreferenceConfig) error {
	rCon, err := client.GetRedisClient()
	if err != nil {
		return resp.RedisConnectError.ErrorDetail(err)
	}
	// 获取数据库连接
	db, err := client.GetDbClient()
	if err != nil {
		return resp.DbConnectError.ErrorDetail(err)
	}
	_, err = db.Table("system_preference_config").Where("id = ?", safeOpts.Id).
		Cols("min_pwd_level", "force_update_state", "pwd_validity", "session_validity", "access_rule_state").
		Update(safeOpts)
	if err != nil {
		conf.Logger.Error("设置安全信息失败!", zap.Error(err))
		return resp.DbUpdateError.ErrorDetail(err)
	}
	go func() {
		rCon.Del(conf.Options.AccessRuleModeKey)
	}()
	return nil
}

// 设置登录配置
func (so *SystemOptionsSvc) SetLoginOptions(loginOpts entity.SystemPreferenceConfig) error {
	// 获取数据库连接
	db, err := client.GetDbClient()
	if err != nil {
		return resp.DbConnectError.ErrorDetail(err)
	}
	_, err = db.Table("system_preference_config").
		Cols("login_config_state", "login_limit_time", "login_pwd_error", "login_lock_time").
		Where("id = ?", loginOpts.Id).Update(loginOpts)
	if err != nil {
		conf.Logger.Error("更新登录配置信息失败", zap.Error(err))
		return resp.DbUpdateError.ErrorDetail(err)
	}
	// 若有账号在锁定当中，管理员将该配置设置为失效，则所有锁定的账号解锁
	go func() {
		if loginOpts.LoginConfigState == 0 {
			so.DeleteLockUser()
		}
	}()
	return nil
}

// 校验License是否过期
func (so *SystemOptionsSvc) CheckLicense(user entity.SystemUserInfo) (*response.CheckLicense, error) {
	systemOptions, e := so.GetSystemOptions()
	if e != nil {
		return nil, e
	}
	var checkLicense = response.CheckLicense{
		SystemOptionsId: systemOptions.Id,
		SystemVersion:   systemOptions.SystemVersion,
		SystemUser:      systemOptions.SystemUser,
		License:         systemOptions.License,
		CanUpdLicense:   2,
		IsExpired:       3,
		DeadDate:        systemOptions.LicenseDeadDate,
	}
	//if user.SystemId == util.AdminSystemUser {
	//	checkLicense.CanUpdLicense = 1
	//}
	if len(checkLicense.DeadDate) == 10 {
		deadDate, _ := time.Parse("2006-01-02", checkLicense.DeadDate)
		// It is shorthand for t.Sub(time.Now()).
		leftDays := time.Until(deadDate).Hours() / 24
		if leftDays <= 0 {
			checkLicense.IsExpired = 1
		} else if int(leftDays) <= systemOptions.LicenseInformDay {
			checkLicense.IsExpired = 2
		}
	}
	return &checkLicense, nil
}

// 所有锁定的账号解锁
func (so *SystemOptionsSvc) DeleteLockUser() (data interface{}, err error) {
	rCon, err := client.GetRedisClient()
	if err != nil {
		return nil, resp.RedisConnectError.ErrorDetail(err)
	}
	ruleKeys, err := rCon.Keys(client.RULEKEY)
	if err != nil {
		conf.Logger.Error("查询锁定规则key序列失败")
		return nil, resp.RedisExecError.ErrorDetail(err)
	}
	lockKeys, err := rCon.Keys(client.LOCKKEY)
	if err != nil {
		conf.Logger.Error("查询锁定key序列失败")
		return nil, resp.RedisExecError.ErrorDetail(err)
	}
	lockKeys = append(lockKeys, ruleKeys...)
	if len(lockKeys) == 0 {
		return "无账号锁定", nil
	}
	err = rCon.DelList(lockKeys)
	if err != nil {
		conf.Logger.Error("批量删除锁定key失败")
		return "", resp.RedisExecError.ErrorDetail(err)
	}
	return "", nil
}

func (so *SystemOptionsSvc) GetLicenseInfo(str entity.LicenseInfo) (data interface{}, err error) {
	return license.DecodeLicense(str.License)
}
