package service

import (
	"errors"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/bean/entity"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/bean/vo/request"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/bean/vo/response"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/common/client"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/common/conf"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/pkg/beagle/jsontime"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/pkg/beagle/resp"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/util"
	"xorm.io/xorm"

	"strconv"
	"time"

	"go.uber.org/zap"
)

const (
	roleTypeNormal    = 0
	roleTypeSystem    = 1
	roleTypeSupertube = 2
)

// CreateSystemRole 创建系统角色
func CreateSystemRole(params request.CreateSystemRoleReq) (resultData int64, err error) {
	roleId := util.GetUUID()
	systemRole := entity.SystemRole{
		RoleName:    params.RoleName,
		RoleDesc:    params.RoleDesc,
		State:       params.State,
		CreatedBy:   params.CreatedBy,
		CreatedTime: jsontime.Time(time.Now()),
		UpdatedTime: jsontime.Time(time.Now()),
		UpdatedBy:   params.CreatedBy,
		RoleType:    roleTypeNormal,
		RoleId:      roleId,
		//DataPurview: params.DataPurview,
	}
	db, err := client.GetDbClient()
	if err != nil {
		err = resp.DbConnectError.ErrorDetail(err)
		return
	}
	// 开始事务
	_, err = db.Transaction(func(session *xorm.Session) (data interface{}, err error) {
		// 插入角色
		_, err = session.Insert(&systemRole)
		if err != nil {
			conf.Logger.Error("创建系统角色失败", zap.Error(err))
			err = resp.DbInsertError.ErrorDetail(errors.New("创建系统角色失败"))
			return
		}
		// 查询菜单权限
		var systemMenuArr []entity.SystemMenu
		err = session.In("id", params.MenuIds).Find(&systemMenuArr)
		if err != nil {
			conf.Logger.Error("查询菜单列表失败", zap.Error(err))
			err = resp.DbSelectError.ErrorDetail(errors.New("查询菜单列表失败"))
			return
		}
		var systemRoleMenuArr []entity.SystemRoleMenu
		currentTime := time.Now()
		for _, v := range systemMenuArr {
			tempSystemRoleMenu := entity.SystemRoleMenu{
				RoleId:      roleId,
				MenuId:      v.MenuId,
				CreatedTime: currentTime,
			}
			systemRoleMenuArr = append(systemRoleMenuArr, tempSystemRoleMenu)
		}
		// 权限菜单不为空则添加角色菜单
		if len(systemRoleMenuArr) > 0 {
			_, err = session.Insert(&systemRoleMenuArr)
			if err != nil {
				conf.Logger.Error("创建系统角色菜单失败", zap.Error(err))
				err = resp.DbInsertError.ErrorDetail(errors.New("创建系统角色菜单失败"))
				return
			}
		}
		return
	})
	if err != nil {
		return
	}
	return
}

// UpdateSystemRole 修改系统角色
func UpdateSystemRole(params request.UpdateSystemRoleReq) (resultData int64, err error) {
	db, err := client.GetDbClient()
	if err != nil {
		err = resp.DbConnectError.ErrorDetail(err)
		return
	}
	systemRole := entity.SystemRole{}
	has, err := db.ID(params.Id).Get(&systemRole)
	if err != nil || !has {
		err = resp.DbSelectError.ErrorDetail(errors.New("角色异常"))
		return
	}

	roleId := systemRole.RoleId
	systemRole = entity.SystemRole{
		RoleName:    params.RoleName,
		RoleDesc:    params.RoleDesc,
		State:       params.State,
		UpdatedTime: jsontime.Time(time.Now()),
		UpdatedBy:   params.UpdatedBy,
		//DataPurview: params.DataPurview,
	}
	// 查询菜单权限
	menuIds := []string{}
	err = db.Table("system_menu").Cols("menu_id").In("id", params.MenuIds).Find(&menuIds)
	if err != nil {
		conf.Logger.Error("查询菜单失败", zap.Error(err))
		err = resp.DbSelectError.ErrorDetail(errors.New("查询菜单失败"))
		return
	}
	// 开启事务
	_, err = db.Transaction(func(session *xorm.Session) (data interface{}, err error) {
		// 内置角色和超管不可编辑其他信息 只能修改菜单
		if systemRole.RoleType != roleTypeSupertube && systemRole.RoleType != roleTypeSystem {
			// 更新角色信息
			_, err = session.Cols("role_name", "role_desc", "state", "updated_time", "updated_by", "data_purview").ID(params.Id).Update(&systemRole)
			if err != nil {
				conf.Logger.Error("修改系统角色失败", zap.Error(err))
				err = resp.DbUpdateError.ErrorDetail(errors.New("修改系统角色失败"))
				return
			}
		}
		// 删除原有菜单权限
		delData := entity.SystemRoleMenu{}
		_, err = session.Where("role_id=?", roleId).Delete(delData)
		if err != nil {
			conf.Logger.Error("删除原有菜单权限失败", zap.Error(err))
			err = resp.DbDeleteError.ErrorDetail(errors.New("删除原有菜单权限失败"))
		}
		systemRoleMenuArr := []entity.SystemRoleMenu{}
		currentTime := time.Now()
		for _, v := range menuIds {
			tempSystemRoleMenu := entity.SystemRoleMenu{
				RoleId:      roleId,
				MenuId:      v,
				CreatedTime: currentTime,
			}
			systemRoleMenuArr = append(systemRoleMenuArr, tempSystemRoleMenu)
		}
		// 权限菜单不为空则添加角色菜单
		if len(systemRoleMenuArr) > 0 {
			_, err = session.Insert(&systemRoleMenuArr)
			if err != nil {
				conf.Logger.Error("创建系统角色菜单失败", zap.Error(err))
				err = resp.DbInsertError.ErrorDetail(errors.New("创建系统角色菜单失败"))
				return
			}
		}
		return
	})
	if err != nil {
		return
	}
	return
}

// SystemRoleState 修改系统角色状态
func SystemRoleState(params request.SystemRoleStateReq) (resultData int64, err error) {
	systemRole := entity.SystemRole{}
	db, err := client.GetDbClient()
	if err != nil {
		err = resp.DbConnectError.ErrorDetail(err)
		return
	}
	id, _ := strconv.Atoi(params.Id)
	get, err := db.ID(id).Get(&systemRole)
	if err != nil || !get {
		conf.Logger.Error("查询角色失败", zap.Error(err))
		err = resp.DbUpdateError.ErrorDetail(errors.New("查询角色失败"))
		return
	}
	if systemRole.RoleType == roleTypeSystem || systemRole.RoleType == roleTypeSupertube {
		conf.Logger.Error("当前角色不可更换状态", zap.Error(err))
		err = resp.DbUpdateError.ErrorDetail(errors.New("当前角色不可更换状态"))
		return
	}
	systemRole.State = 0
	if params.State == "1" {
		systemRole.State = 1
	}
	obj := db.ID(id)
	update, err := obj.Cols("state").Where("is_deleted = 0").Update(&systemRole)
	if err != nil {
		conf.Logger.Error("修改系统角色状态失败", zap.Error(err))
		err = resp.DbUpdateError.ErrorDetail(errors.New("修改系统角色状态失败"))
		return
	}
	return update, nil
}

// SystemRoleList 获取系统角色列表
func SystemRoleList(params *request.SystemRoleListReq) (resultData []response.SystemRoleListRes, count int64, err error) {
	systemRoleArr := []response.SystemRoleList{}
	db, err := client.GetDbClient()
	if err != nil {
		err = resp.DbConnectError.ErrorDetail(err)
		return
	}
	modelObj := db.Table("system_role as sr")
	modelObj.Select("sr.id,sr.role_name,sr.created_time,sr.created_by,sr.role_desc,sr.state,sr.role_type,sr.role_id,su.system_account")
	modelObj.Join("left", "system_user as su", "sr.created_by = su.id")
	if params.Search != "" {
		modelObj.Where("sr.role_name like ?", "%"+params.Search+"%")
	}
	modelObj.Where("sr.is_deleted = 0")
	//if params.IsAdmin != 1 {
	//	modelObj.Where("sr.role_type !=?", roleTypeSupertube)
	//}
	// 查询系统角色列表
	count, err = modelObj.OrderBy("sr.id desc").Limit(params.GetPageSize(), (params.GetPage()-1)*params.GetPageSize()).FindAndCount(&systemRoleArr)
	if err != nil {
		conf.Logger.Error("查询系统角色列表失败", zap.Error(err))
		err = resp.DbSelectError.ErrorDetail(errors.New("查询系统角色列表失败"))
		return
	}

	// 拼装返回结果
	SystemRoleListArr := []response.SystemRoleListRes{}
	roleIds := []string{}
	for _, v := range systemRoleArr {
		SystemRoleListRes := response.SystemRoleListRes{
			Id:          v.Id,
			RoleName:    v.RoleName,
			RoleDesc:    v.RoleDesc,
			State:       v.State,
			CreatedBy:   v.SystemAccount,
			CreatedTime: jsontime.Time(v.CreatedTime),
			RoleType:    v.RoleType,
			RoleId:      v.RoleId,
			//DataPurview: v.DataPurview,
		}
		// 业务系统角色不可分配用户
		//if v.RoleType != 0 {
		//	SystemRoleListRes.CantAllot = 1
		//}
		SystemRoleListArr = append(SystemRoleListArr, SystemRoleListRes)
		roleIds = append(roleIds, v.RoleId)
	}
	// 查询角色关联用户数量
	userCountMap := make([]struct {
		RoleId    string `json:"role_id"`
		UserCount int    `json:"user_count"`
	}, 0)
	err = db.Select("role_id::varchar,count(*) as user_count").Table("system_user_role").In("role_id", roleIds).GroupBy("role_id").Find(&userCountMap)
	if err != nil {
		conf.Logger.Error("查询角色关联用户数量失败", zap.Error(err))
		err = resp.DbSelectError.ErrorDetail(errors.New("查询角色关联用户数量失败"))
		return
	}
	// 用户数关联到角色id
	userCountMapKey := make(map[string]int)
	for _, v := range userCountMap {
		userCountMapKey[v.RoleId] = v.UserCount
	}
	// 用户数拼接到返回结果
	for k, v := range SystemRoleListArr {
		SystemRoleListArr[k].UserCount = userCountMapKey[v.RoleId]
	}
	resultData = SystemRoleListArr
	return resultData, count, nil
}

// SystemAllotUserList 角色可分配账户列表
func SystemAllotUserList(params *request.SystemAllotUserListReq) (resultData []response.SystemAllotUserListRes, count int64, err error) {
	// 账户类型筛选条件不可为业务系统账户，强制清空账户类型筛选项
	//if params.IsAdmin == isAdminSystem {
	//	params.IsAdmin = 0
	//}
	var SystemUserArr []response.SystemAllotUserList
	db, err := client.GetDbClient()
	if err != nil {
		err = resp.DbConnectError.ErrorDetail(err)
		return
	}
	systemRole := entity.SystemRole{}
	get, roleErr := db.Table("system_role").ID(params.RoleId).Get(&systemRole)
	if roleErr != nil || !get {
		conf.Logger.Error("查询角色失败", zap.Error(err))
		err = resp.DbSelectError.ErrorDetail(errors.New("查询角色失败"))
		return
	}

	modelObj := db.Table("system_user as su")
	modelObj.Select("su.id,su.system_account,su.phone,su.organization_id,so.name,su.created_time,su.state,ur.role_id,ur.user_id as ur_user_id")
	modelObj.Join("left", "system_organization as so", "su.organization_id = so.organization_id")
	modelObj.Join("left", "system_user_role as ur", "su.id = ur.user_id and ur.role_id=?", systemRole.RoleId)

	if params.Search != "" {
		params.Search = "%" + params.Search + "%"
		modelObj.Where("su.system_account like ? or su.phone ?  ", params.Search, params.Search)
	}

	if systemRole.RoleType == 1 {
		modelObj.Where("so.data_type < 2")
	}
	if systemRole.RoleType == 2 {
		modelObj.Where("so.data_type = 2")
	}

	if params.OrganizationId != 0 {
		systemOrganization := entity.SystemOrganization{}
		has, organizationErr := db.Table("system_organization").ID(params.OrganizationId).Get(&systemOrganization)
		if organizationErr != nil || !has {
			conf.Logger.Error("查询组织架构失败", zap.Error(err))
			err = resp.DbSelectError.ErrorDetail(errors.New("查询组织架构失败"))
			return
		}
		modelObj.Where("su.organization_id = ?", systemOrganization.OrganizationId)
		if systemRole.RoleType == 2 && systemOrganization.DataType != 2 {
			return nil, 0, nil
		}
	}
	// 根据用户当前角色排除可分配角色类型
	//switch util.RoleId(systemRole.RoleId) {
	//case util.YEWUXITONGJUESE: // 业务系统账户不可分配任何角色的账户
	//	return
	//case util.ZUZHIGUANLIYUAN: //  组织管理员不可分配任何角色的账户
	//	return
	//case util.CHAOJIGUANLIYUAN:
	//	modelObj.Where("su.is_admin = ?", isAdminOrdinary)
	//case util.PINGTAIYUNYINGZHE:
	//	modelObj.Where("su.is_admin = ?", isAdminOrdinary)
	//default:
	//	modelObj.In("su.is_admin", []int{isAdminOrganize, isAdminOrdinary})
	//}
	//
	//// 排除业务系统用户
	//modelObj.Where("su.is_admin != ?", isAdminSystem)
	//if params.IsAdmin != 0 {
	//	modelObj.Where("su.is_admin = ?", params.IsAdmin)
	//}
	modelObj.Where("su.is_deleted = 0")
	count, err = modelObj.OrderBy("su.id desc").Limit(params.GetPageSize(), (params.GetPage()-1)*params.GetPageSize()).FindAndCount(&SystemUserArr)
	if err != nil {
		conf.Logger.Error("查询系统账户列表失败", zap.Error(err))
		err = resp.DbSelectError.ErrorDetail(errors.New("查询系统账户列表失败"))
		return
	}
	var SystemUserListResArr []response.SystemAllotUserListRes
	for _, v := range SystemUserArr {
		SystemUserListRes := response.SystemAllotUserListRes{
			Id:               v.Id,
			SystemAccount:    v.SystemAccount,
			IsAdmin:          v.IsAdmin,
			Phone:            v.Phone,
			OrganizationId:   v.OrganizationId,
			OrganizationName: v.OrganizationName,
			State:            v.State,
			CreatedTime:      jsontime.Time(v.CreatedTime),
			//SystemId:         v.SystemId,
			IsBind: 0,
		}
		if v.UrUserId != 0 {
			SystemUserListRes.IsBind = 1
		}
		SystemUserListResArr = append(SystemUserListResArr, SystemUserListRes)
	}
	resultData = SystemUserListResArr
	return resultData, count, nil
}

// SystemRoleDetail 系统角色详情
func SystemRoleDetail(params request.SystemRoleDetailReq) (resultData response.SystemRoleDetailRes, err error) {
	systemRole := entity.SystemRole{}
	systemRoleDetailRes := response.SystemRoleDetailRes{}
	db, err := client.GetDbClient()
	if err != nil {
		err = resp.DbConnectError.ErrorDetail(err)
		return
	}
	// 查询系统角色详情
	id, _ := strconv.Atoi(params.Id)
	has, err := db.ID(id).Where("is_deleted = 0").Get(&systemRole)
	if err != nil || !has {
		conf.Logger.Error("查询系统角色详情失败", zap.Error(err))
		err = resp.DbSelectError.ErrorDetail(errors.New("查询系统角色详情失败"))
		return
	}
	// 拼装返回结果
	systemRoleDetailRes.Id = systemRole.Id
	systemRoleDetailRes.RoleId = systemRole.RoleId
	systemRoleDetailRes.RoleName = systemRole.RoleName
	systemRoleDetailRes.RoleDesc = systemRole.RoleDesc
	systemRoleDetailRes.State = systemRole.State
	//systemRoleDetailRes.DataPurview = systemRole.DataPurview
	systemRoleDetailRes.RoleId = systemRole.RoleId
	systemRoleDetailRes.CreatedTime = jsontime.Time(systemRole.CreatedTime)
	systemRoleDetailRes.UpdatedTime = jsontime.Time(systemRole.UpdatedTime)
	systemRoleDetailRes.RoleType = systemRole.RoleType

	// 查询角色关联菜单
	systemMenusArr := []response.SystemMenus{}
	err = db.Select("sm.id,sm.menu_id").Table("system_role_menu as rm").
		Join("left", "system_menu sm", "rm.menu_id = sm.menu_id").
		Where("rm.role_id=?", systemRole.RoleId).
		Find(&systemMenusArr)
	if err != nil {
		conf.Logger.Error("查询角色菜单失败", zap.Error(err))
		err = resp.DbSelectError.ErrorDetail(errors.New("查询角色菜单失败"))
		return
	}
	systemRoleDetailRes.Menus = systemMenusArr
	resultData = systemRoleDetailRes
	return resultData, nil
}

// SystemRoleAllotmentUser 分配用户
func SystemRoleAllotmentUser(params *request.SystemRoleAllotmentUserReq) (err error) {
	db, err := client.GetDbClient()
	if err != nil {
		err = resp.DbConnectError.ErrorDetail(err)
		return
	}
	// 查询角色
	systemRole := entity.SystemRole{}
	_, err = db.Table("system_role").ID(params.Id).Where("is_deleted =0").Get(&systemRole)
	if err != nil {
		conf.Logger.Error("查询角色失败", zap.Error(err))
		err = resp.DbSelectError.ErrorDetail(errors.New("查询角色失败"))
		return err
	}
	//if systemRole.RoleId == string(util.YEWUXITONGJUESE) || systemRole.RoleId == string(util.ZUZHIGUANLIYUAN) {
	//	err = res.DbSelectError.ErrorDetail(errors.New("当前角色不可分配用户"))
	//	return err
	//}
	// 开启事务
	_, err = db.Transaction(func(session *xorm.Session) (data interface{}, err error) {
		// 删除原有角色关联记录

		delData := entity.SystemUserRole{}
		_, err = session.Where("role_id=?", systemRole.RoleId).Delete(delData)
		if err != nil {
			conf.Logger.Error("删除原有角色关联失败", zap.Error(err))
			err = resp.DbDeleteError.ErrorDetail(errors.New("删除原有角色关联失败"))
		}
		// 查询用户信息
		systemUserArr := []entity.SystemUser{}
		//userCount, err := session.In("id", params.UserIds).Where("is_admin !=?", isAdminSystem).FindAndCount(&systemUserArr)
		userCount, err := session.In("id", params.UserIds).FindAndCount(&systemUserArr)
		if err != nil {
			conf.Logger.Error("查询用户失败", zap.Error(err))
			err = resp.DbSelectError.ErrorDetail(errors.New("查询用户失败"))
			return
		}
		// 查询结果用户数和入参用户数不匹配则认定为有异常用户
		if len(params.UserIds) != int(userCount) {
			err = resp.DbSelectError.ErrorDetail(errors.New("无法分配，存在异常用户！"))
			return
		}
		systemUserRoleArr := []entity.SystemUserRole{}
		for _, v := range systemUserArr {
			temp := entity.SystemUserRole{
				UserId:      v.Id,
				RoleId:      systemRole.RoleId,
				CreatedTime: time.Now(),
			}
			systemUserRoleArr = append(systemUserRoleArr, temp)
		}
		if len(systemUserRoleArr) > 0 {
			// 添加角色和用户关联记录
			_, err = session.Insert(systemUserRoleArr)
			if err != nil {
				conf.Logger.Error("分配用户失败", zap.Error(err))
				err = resp.DbUpdateError.ErrorDetail(errors.New("分配用户失败"))
				return
			}
		}

		return
	})
	return
}

// DeleteSystemRole 删除系统角色
func DeleteSystemRole(params request.DeleteSystemRoleReq) (err error) {
	db, err := client.GetDbClient()
	if err != nil {
		err = resp.DbConnectError.ErrorDetail(err)
		return
	}
	roleIds := []string{}
	SystemUserCount, err := db.Table("system_role").Cols("role_id").In("id", params.Ids).Where("state!=1 and role_type !=1 and role_type !=2").FindAndCount(&roleIds)
	if err != nil {
		conf.Logger.Error("查询需要删除的角色失败", zap.Error(err))
		err = resp.DbSelectError.ErrorDetail(errors.New("查询需要删除的角色失败"))
		return err
	}
	if len(params.Ids) != int(SystemUserCount) {
		err = resp.DbSelectError.ErrorDetail(errors.New("删除内容中有未禁用或内置角色，不可删除！"))
		return
	}
	// 开启事务
	_, err = db.Transaction(func(session *xorm.Session) (data interface{}, err error) {
		// 删除系统角色
		updateData := entity.SystemRole{}
		updateData.IsDeleted = 1
		_, err = session.Cols("is_deleted").In("role_id", roleIds).Update(&updateData)
		if err != nil {
			conf.Logger.Error("删除系统角色失败", zap.Error(err))
			err = resp.DbDeleteError.ErrorDetail(errors.New("删除系统角色失败"))
			return
		}
		// 删除角色对应菜单
		delData := entity.SystemRoleMenu{}
		_, err = session.In("role_id", roleIds).Delete(delData)
		if err != nil {
			conf.Logger.Error("删除角色菜单权限失败", zap.Error(err))
			err = resp.DbDeleteError.ErrorDetail(errors.New("删除角色菜单权限失败"))
		}
		return
	})
	return
}
