package service

import (
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/bean/entity"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/bean/vo/request"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/bean/vo/response"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/common/client"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/common/conf"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/pkg/beagle/jsontime"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/pkg/beagle/resp"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/util"
	"go.uber.org/zap"
	"xorm.io/xorm"

	"github.com/gin-gonic/gin"
	"github.com/pkg/errors"

	"time"
)

type Dict struct {
	Ctx  *gin.Context
	User entity.SystemUserInfo
}

func (c *Dict) Check(id, parentId string, name string) (err error) {

	db, err := client.GetDbClient()
	if err != nil {
		err = resp.DbConnectError.ErrorDetail(err)
		return
	}
	session := db.NewSession()
	session.Table("dict")
	session.Where("parent_id = ? and name = ? and is_delete = 0", parentId, name)
	if id != "" {
		session.Where("id != ?", id)
	}

	cnt, err := session.Count(&entity.Dict{})
	if err != nil {
		err = errors.Wrap(err, "数据库查重失败！")
		return
	}

	if cnt > 0 {
		if id != "" {
			err = errors.New("编辑失败，已存在相同的字典标签！")
		} else {
			err = errors.New("新增失败，已存在相同的字典标签！")
		}
		return
	}

	return
}

func (c *Dict) List(req request.DictReq) (dictListRes []*response.DictListRes, count int64, err error) {

	db, err := client.GetDbClient()
	if err != nil {
		err = resp.DbConnectError.WithError(err)
		return dictListRes, 0, err
	}

	session := db.NewSession()
	session.Select("id,class,name,parent_id,status,min_val,max_val,sort,description,updated_at,unit,built_in").Table("dict")
	session.Where("is_delete = 0 ")

	if req.Id != "" {
		session.Where("id = ? ", req.Id)
	}
	if req.ParentId != "" {
		session.Where("parent_id = ? ", req.ParentId)
	}
	if req.Class != 0 {
		session.Where("class = ? ", req.Class)
	}
	if req.Status != 0 {
		session.Where("status = ? ", req.Status)
	}
	if req.Search != "" {
		keyword := util.SpecialEscape(req.Search)
		session.Where("name like ?", "%"+keyword+"%")
	}

	count, err = session.OrderBy(" class,sort").Limit(req.GetPageSize(), (req.GetPage()-1)*req.GetPageSize()).
		FindAndCount(&dictListRes)
	if err != nil {
		return dictListRes, 0, resp.DbSelectError.WithError(err)
	}
	return
}

func (c *Dict) DictTree(req request.DictTreeReq) (componentDictTreeRes []*response.DictListRes, err error) {
	listReq := request.DictReq{
		Class:    req.Class,
		ParentId: req.ParentId,
	}
	componentDictList, _, err := c.List(listReq)
	if err != nil {
		return
	}

	componentDictTreeRes = getTreeIterative(componentDictList, req.ParentId)

	return
}

func getTreeIterative(list []*response.DictListRes, parentId string) []*response.DictListRes {
	memo := make(map[string]*response.DictListRes)
	for _, v := range list {
		if _, ok := memo[v.Id]; ok {
			v.Children = memo[v.Id].Children
			memo[v.Id] = v
		} else {
			v.Children = make([]*response.DictListRes, 0)
			memo[v.Id] = v
		}
		if _, ok := memo[v.ParentId]; ok {
			memo[v.ParentId].Children = append(memo[v.ParentId].Children, memo[v.Id])
		} else {
			memo[v.ParentId] = &response.DictListRes{Children: []*response.DictListRes{memo[v.Id]}}
		}
	}
	return memo[parentId].Children

}

func (c *Dict) Add(req request.AddDictReq) (err error) {

	//检查重复
	err = c.Check("", req.ParentId, req.Name)
	if err != nil {
		return
	}

	dict := &entity.Dict{
		Id:          util.GetUUID(),
		Class:       req.Class,
		Name:        req.Name,
		ParentId:    req.ParentId,
		Status:      req.Status,
		Description: req.Description,
		CreatedBy:   c.User.Id,
		CreatedAt:   jsontime.Time(time.Now()),
		UpdatedBy:   c.User.Id,
		UpdatedAt:   jsontime.Time(time.Now()),
		MinVal:      req.MinVal,
		MaxVal:      req.MaxVal,
		Unit:        req.Unit,
	}

	db, err := client.GetDbClient()
	if err != nil {
		err = resp.DbConnectError.ErrorDetail(err)
		return
	}

	_, err = db.Insert(dict)
	if err != nil {
		err = resp.DbInsertError.WithData(err)
		return
	}

	return

}

func (c *Dict) Update(req request.UpdateDictReq) (err error) {

	//检查重复
	err = c.Check(req.Id, req.ParentId, req.Name)
	if err != nil {
		return
	}

	db, err := client.GetDbClient()
	if err != nil {
		err = resp.DbConnectError.ErrorDetail(err)
		return
	}

	session := db.NewSession()
	req.UpdatedBy = c.User.Id
	req.UpdatedAt = jsontime.Time(time.Now())

	_, err = session.Table("dict").
		Cols("name,status,description,updated_by,updated_at,min_val,max_val,unit").
		Where("id = ? and is_delete = 0", req.Id).Update(&req)
	if err != nil {
		err = errors.Wrap(err, "修改字典失败！")
		return
	}
	return

}

func (c *Dict) Del(id string) (err error) {

	db, err := client.GetDbClient()
	if err != nil {
		err = resp.DbConnectError.ErrorDetail(err)
		return
	}

	session := db.NewSession()

	_, err = session.Table("dict").
		Cols("is_delete").
		Where("id = ? ", id).Update(map[string]int{
		"is_delete": 1,
	})
	if err != nil {
		err = errors.Wrap(err, "删除字典失败！")
		return
	}
	return

}

func (c *Dict) ClassList(className string) (classList []response.ClassListRes, err error) {

	db, err := client.GetDbClient()
	if err != nil {
		err = resp.DbConnectError.ErrorDetail(err)
		return
	}

	session := db.NewSession()

	if className != "" {
		session.Where("class_name like ?", "%"+className+"%")
	}

	session.OrderBy("class_id")

	//执行查询
	err = session.Table("dict_class").Find(&classList)
	if err != nil {
		err = errors.Wrap(err, "查询字典分类列表失败")
		return
	}
	return
}

func (c *Dict) DictSort(req []request.DictSortInput) error {
	db, err := client.GetDbClient()
	if err != nil {
		return resp.DbConnectError.ErrorDetail(err)
	}

	_, err = db.Transaction(func(x *xorm.Session) (interface{}, error) {
		for _, v := range req {
			_, err := x.Table("dict").Cols("sort, updated_at, updated_by").Where("id = ? and is_delete = 0", v.Id).Update(
				&entity.Dict{
					Sort:      v.Sort,
					UpdatedBy: c.User.Id,
					UpdatedAt: jsontime.Time(time.Now()),
				})
			if err != nil {
				conf.Logger.Error("排序失败", zap.Error(err))
				return nil, resp.DbUpdateError.ErrorDetail(err)
			}
		}
		return nil, nil
	})

	return err
}

func (d *Dict) UpdateStatus(id string, status string) (total int64, err error) {
	db, err := client.GetDbClient()
	if err != nil {
		return 0, resp.DbConnectError.ErrorDetail(err)
	}

	total, err = db.Table("dict").Where("id = ? and is_delete = 0 ", id).Update(map[string]interface{}{
		"status":     status,
		"updated_by": d.User.Id,
	})
	if err != nil {
		conf.Logger.Error("Update UpdateStatus ", zap.String("erro", err.Error()))
		return 0, resp.DbUpdateError.ErrorDetail(err)
	}
	return
}

//func (c *Dict) ManageList(req request.DictManageListReq) (componentDictTreeRes []*response.DictListRes, err error) {
//	listReq := request.DictReq{
//		Class:    req.Class,
//		ParentId: -1,
//	}
//	componentDictList, err := c.list(listReq)
//	if err != nil {
//		return
//	}
//
//	componentDictTreeRes = getTreeIterative(componentDictList, req.ParentId)
//
//	return
//}

//func (c *Dict) list(req request.DictReq) (componentDictListRes []*response.DictListRes, err error) {
//
//	finder := zorm.NewFinder().Append("select id,class,name,parent_id from " + new(entity.Dict).GetTableName())
//	finder.Append("WHERE status = 0 AND is_delete = 0 ")
//
//	if req.Id != 0 {
//		finder.Append(" AND id = ? ", req.Id)
//	}
//	if req.ParentId != -1 {
//		finder.Append(" AND parent_id = ? ", req.ParentId)
//	}
//	if req.Class != 0 {
//		finder.Append(" AND class = ? ", req.Class)
//	}
//
//	finder.Append("order by name")
//	//执行查询
//	err = zorm.Query(c.Ctx, finder, &componentDictListRes, nil)
//	if err != nil {
//		err = errors.Wrap(err, "查询字典列表失败")
//		return
//	}
//	return
//}
