package util

import (
	"fmt"
	"github.com/gin-gonic/gin/binding"
	"github.com/go-playground/locales/zh"
	ut "github.com/go-playground/universal-translator"
	"github.com/go-playground/validator/v10"
	translations "github.com/go-playground/validator/v10/translations/zh"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/common/conf"
	"go.uber.org/zap"
	"reflect"
	"regexp"
)

var (
	validate *validator.Validate
	Trans    ut.Translator
)

func RegValid() {
	// 注册自定义的校验器错误消息
	Trans, _ = ut.New(zh.New()).GetTranslator("zh")
	validate = binding.Validator.Engine().(*validator.Validate)
	if err := translations.RegisterDefaultTranslations(validate, Trans); err != nil {
		fmt.Println("RegisterDefaultTranslations failed", err)
		return
	}

	validate.RegisterTagNameFunc(func(fld reflect.StructField) string {
		if jsonTag := fld.Tag.Get("json"); jsonTag != "" {
			return jsonTag
		}
		if formTag := fld.Tag.Get("form"); formTag != "" {
			return formTag
		}
		return fld.Name
	})

	err := validate.RegisterValidation("phone", validatePhone)
	if err != nil {
		conf.Logger.Info("validate register failed", zap.Error(err))
	}
	setTranslation(validate, Trans)
}

// 校验手机号
func validatePhone(fl validator.FieldLevel) bool {
	phone := fl.Field().String()
	pattern := "^((13[0-9])|(14[5,7])|(15[0-3,5-9])|(17[0,3,5-8])|(18[0-9])|166|198|199|(147))\\d{8}$"
	regex, err := regexp.Compile(pattern)
	if err != nil {
		return false
	}
	isValid := regex.MatchString(phone)
	return isValid
}

// 中文翻译
func setTranslation(validate *validator.Validate, trans ut.Translator) {
	validate.RegisterTranslation(
		"phone",
		trans,
		func(ut ut.Translator) error {
			return ut.Add("phone", "{0}不是一个有效的手机号码！", true)
		},
		func(ut ut.Translator, fe validator.FieldError) string {
			return fmt.Sprintf("%s不是一个有效的手机号码！", fe.Field())
		},
	)
}
