package response

import (
	"fmt"
	json "github.com/json-iterator/go"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/bean/entity"
	"sort"
	"testing"
)

func BuildTree(nodes []entity.AlertClass) ([]*AlertClassNode, error) {
	nodeMap := make(map[int]*AlertClassNode)

	// 创建所有节点并存储到映射表中
	for _, node := range nodes {
		tree := &AlertClassNode{
			AlertClass: node,
			Children:   []*AlertClassNode{},
		}
		nodeMap[node.ClassId] = tree
	}

	var rootNodes []*AlertClassNode
	for _, node := range nodes {
		if node.ParentId == 0 {
			rootNodes = append(rootNodes, nodeMap[node.ClassId])
		} else {
			parent := nodeMap[node.ParentId]
			if parent == nil {
				return nil, fmt.Errorf("parent node not found for ClassId: %d", node.ClassId)
			}
			parent.Children = append(parent.Children, nodeMap[node.ClassId])
		}
	}
	sortTree(rootNodes)

	return rootNodes, nil
}

func sortTree(nodes []*AlertClassNode) {
	sort.Slice(nodes, func(i, j int) bool {
		return nodes[i].SortOrder < nodes[j].SortOrder
	})
	for _, node := range nodes {
		sortTree(node.Children)
	}
}

func TestTree(t *testing.T) {
	// 示例数据
	data := []entity.AlertClass{
		{ClassId: 1, ClassName: "Root", ParentId: 0, SortOrder: 0},
		{ClassId: 2, ClassName: "Child 1", ParentId: 1, SortOrder: 0},
		{ClassId: 3, ClassName: "Child 2", ParentId: 1, SortOrder: 0},
		{ClassId: 4, ClassName: "Grandchild 3", ParentId: 2, SortOrder: 3},
		{ClassId: 5, ClassName: "Grandchild 1", ParentId: 2, SortOrder: 1},
		{ClassId: 6, ClassName: "Grandchild 2", ParentId: 2, SortOrder: 2},
	}

	rootNodes, err := BuildTree(data)
	if err != nil {
		fmt.Println("Failed to build tree:", err)
		return
	}

	// 将树形结构转换为 JSON 字符串
	jsonData, err := json.Marshal(rootNodes)
	if err != nil {
		fmt.Println("Failed to marshal tree:", err)
		return
	}

	fmt.Println(string(jsonData))
}
