package main

import (
	"fmt"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/service/cron"
	"net/http"
	"os"
	"time"

	"github.com/gin-gonic/gin"
	"github.com/spf13/pflag"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/common/client"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/common/conf"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/pkg/beagle/log"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/router"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/util"
	"go.uber.org/zap"
	"go.uber.org/zap/zapcore"
)

// run args
var (
	port   = pflag.Int("port", 80, "server port")
	prefix = pflag.String("prefix", "/v1/api", "url prefix")
)

// main start
func main() {
	initTimeZone()
	// init ansible hosts
	initAnsibleHosts()
	pflag.Parse() // init start args
	initConfig()
	// init log config
	cfg := initLogConfig()
	conf.Logger = log.NewLogger(cfg, conf.LoggerLevel)
	// init validator
	util.RegValid()
	// db client
	go client.GetDbClient()
	// redis client
	go client.GetRedisClient()

	//启动定时任务
	cron.StartCron()

	// server start...
	conf.Logger.Error("server start err", zap.Error(newServer().ListenAndServe()))
}

// init commonutil config value
func initConfig() {
	conf.LoggerLevel = util.SetEnvStr("LOG_LEVEL", "info")
	conf.RunMode = util.SetEnvStr("GIN_MODE", "debug") // project run mode, available parameters: debug、release
	conf.Options = &conf.Config{
		Prefix:         *prefix,
		DbURL:          util.SetEnvStr("DB_URL", "host=localhost port=9010 user=postgres password=passwd123 dbname=postgres sslmode=disable"),
		DbDriver:       util.SetEnvStr("DB_DRIVER", "postgres"),
		RedisURL:       util.SetEnvStr("REDIS_URL", "localhost:7001"),
		RedisDB:        0,
		RedisTag:       "bg",
		LogDirPrefix:   util.SetEnvStr("LOG_DIR_PREFIX", "/app/log"),                         // 日志目录
		LogDirName:     util.SetEnvStr("LOG_NAME", "syslog"),                                 // 日志名称
		LogSaveDays:    util.SetEnvInt("LOG_SAVE_DAYS", 7),                                   // 日志最大存储天数
		LogMode:        util.SetEnvInt("LOG_MODE", 1),                                        // 1.标准打印 2.输出文件
		ArgBool:        util.SetEnvBool("ARG_BOOL", false),                                   // 示例参数
		ArgInt:         util.SetEnvInt("ARG_INT", 10),                                        // 示例参数
		MinioServer:    util.SetEnvStr("MINIO_SERVER", "https://cache.wodcloud.com"),         // Minio 服务地址
		MinioAccessKey: util.SetEnvStr("MINIO_ACCESS_KEY", "beagleadmin"),                    // Minio Access Key
		MinioSecretKey: util.SetEnvStr("MINIO_SECRET_KEY", "H76cPmwvH7vJ"),                   // Minio Secret
		MinioBucket:    util.SetEnvStr("MINIO_BUCKET", "so-operation"),                       // Minio Bucket
		TempDirPrefix:  util.SetEnvStr("TEMP_DIR_PREFIX", "/app/xlsx/"),                      //模板目录前缀
		PrometheusHost: util.SetEnvStr("PROMETHEUS_HOST", "https://prometheus.wodcloud.com"), // Prometheus Host
	}
}

// init log config
func initLogConfig() zapcore.EncoderConfig {
	return zapcore.EncoderConfig{
		LevelKey:       "level",
		TimeKey:        "time",
		MessageKey:     "msg",
		NameKey:        "logger",
		CallerKey:      "caller",
		StacktraceKey:  "stacktrace",
		LineEnding:     zapcore.DefaultLineEnding,
		EncodeLevel:    zapcore.CapitalLevelEncoder,
		EncodeTime:     zapcore.TimeEncoderOfLayout("2006-01-02 15:04:05"),
		EncodeDuration: zapcore.SecondsDurationEncoder,
		EncodeCaller:   zapcore.ShortCallerEncoder,
	}
}

func newServer() *http.Server {
	// set run mod
	gin.SetMode(conf.RunMode)
	// load gin router
	r := gin.New()
	router.Load(r, log.GinLogger(), log.GinRecovery(true))
	conf.Logger.Info("server is starting...", zap.Int("port", *port))
	return &http.Server{
		Addr:           fmt.Sprintf(":%d", *port),
		Handler:        r,
		ReadTimeout:    30 * time.Second,
		WriteTimeout:   30 * time.Second,
		MaxHeaderBytes: 1 << 20,
	}
}

func initTimeZone() {
	//var cstZone, _ = time.LoadLocation("Asia/Shanghai")
	var cstZone = time.FixedZone("CST", 8*3600) // 固定东八区(CST: China Standard Time)
	time.Local = cstZone
}

func initAnsibleHosts() {
	err := os.Mkdir("/etc/ansible", os.ModePerm)
	if err != nil {
		fmt.Println("创建目录失败!")
	}
	f, err := os.Create("/etc/ansible/hosts")
	defer f.Close()
	if err != nil {
		fmt.Println(err.Error())
	} else {
		_, err := f.Write([]byte("[web]\n"))
		if err != nil {
			fmt.Println(err.Error())
		}
	}
}
