package jsontime

import (
	"bytes"
	"database/sql/driver"
	"fmt"
	"strings"
	"time"
)

// 时间序列化和反序列化

var CSTZone = time.FixedZone("CST", 8*3600) // 固定东八区(CST: China Standard Time)

// Time 自定义时间
// 设置当前时间: jsonDate := Time(time.Now())
type Time time.Time

func Now() Time {
	return Time(time.Now())
}

// MarshalJSON JsonDate序列化
func (t Time) MarshalJSON() ([]byte, error) {
	b := make([]byte, 0, len("2006-01-02 15:04:05")+2)
	b = append(b, '"')
	b = time.Time(t).AppendFormat(b, "2006-01-02 15:04:05")
	b = append(b, '"')
	//b = bytes.Trim(b, "\"")
	if bytes.Contains(b, []byte("0001-01-01 00:00:00")) {
		b = []byte("\"\"")
	}
	return b, nil
}

// UnmarshalJSON JsonDate反序列化
func (t *Time) UnmarshalJSON(data []byte) error {
	s := string(data)
	s = strings.Trim(s, "\"")

	if s == "" {
		*t = Time(time.Time{})
		return nil
	}

	layouts := []string{
		"2006-01-02 15:04:05",
		"2006-01-02 15:04:05.000",
		"2006-01-02 15:04:05.000000",
		"2006-01-02T15:04:05Z",
		"2006-01-02T15:04:05.000Z",
		"2006-01-02T15:04:05.000000Z",
	}

	var err error
	var jDate time.Time
	for _, layout := range layouts {
		jDate, err = time.ParseInLocation(layout, s, time.Local)
		if err == nil {
			break
		}
	}

	if err != nil {
		return err
	}

	*t = Time(jDate)
	return nil
}

func (t *Time) String() string {
	stamp := time.Time(*t).Format("2006-01-02 15:04:05")
	if stamp == "0001-01-01 00:00:00" {
		return ""
	}
	return stamp
}

// Value 实现 driver.Valuer 接口
func (t Time) Value() (driver.Value, error) {
	return time.Time(t), nil
}

// Scan 实现 sql.Scanner 接口
func (t *Time) Scan(value interface{}) error {
	switch v := value.(type) {
	case time.Time:
		*t = Time(v)
	case []byte:
		jTime, err := time.ParseInLocation("2006-01-02 15:04:05", string(v), time.Local)
		if err != nil {
			return err
		}
		*t = Time(jTime)
	case string:
		jTime, err := time.ParseInLocation("2006-01-02 15:04:05", v, time.Local)
		if err != nil {
			return err
		}
		*t = Time(jTime)
	case nil:
		*t = Time(time.Time{})
	default:
		return fmt.Errorf("Time.Scan: unsupported type %T", value)
	}
	return nil
}
