package resp

import (
	"fmt"
	"github.com/go-playground/validator/v10"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/util"
	"strings"

	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/common/conf"
	"go.uber.org/zap"
)

type Resp struct {
	Code int         `json:"code"`
	Msg  string      `json:"msg"`
	Data interface{} `json:"data,omitempty"`
}

func (r Resp) WithMsg(msg string) Resp {
	r.Msg = msg
	return r
}

func (r Resp) WithMsgF(msg string, a ...interface{}) Resp {
	r.Msg = fmt.Sprintf(msg, a)
	return r
}

func (r Resp) WithData(data interface{}) Resp {
	r.Data = data
	return r
}

func (r Resp) WithError(err error) Resp {
	if err != nil {
		r.Data = err.Error()
	}
	return r
}

func (r Resp) ErrorDetail(err error) Resp {
	if err != nil {
		r.Data = err.Error()
	}
	return r
}

// TranslateError 翻译validate验证错误
func (r Resp) TranslateError(err error) Resp {
	translatedErrors := make([]string, 0)
	if validationErrors, ok := err.(validator.ValidationErrors); ok {
		for _, validationError := range validationErrors {
			translatedErrors = append(translatedErrors, validationError.Translate(util.Trans))
		}
	} else {
		translatedErrors = append(translatedErrors, err.Error())
	}
	r.Msg = strings.Join(translatedErrors, ",")
	return r
}

func (r Resp) Error() string {
	return fmt.Sprintf("code:%d Message:%s", r.Code, r.Msg)
}

func DecodeErr(err error, data interface{}) (int, string, interface{}) {
	if err == nil {
		return OK.Code, OK.Msg, data
	}

	switch typed := err.(type) {
	case Resp:
		if typed.Code == 200 {
			return OK.Code, OK.Msg, data
		}
		return typed.Code, typed.Msg, typed.Data
	case *Resp:
		return typed.Code, typed.Msg, typed.Data
	}
	conf.Logger.Error("响应错误信息被拦截", zap.Error(err))
	return Unauthorized.Code, Unauthorized.Msg, err.Error()
}

type BgPageRes struct {
	Code  int         `json:"code"`
	Msg   string      `json:"msg"`
	Data  interface{} `json:"data"`
	Total int64       `json:"total"`
}

func (b BgPageRes) ResData(data interface{}, total int64) BgPageRes {
	if data != nil {
		b.Data = data
	}
	if total != 0 {
		b.Total = total
	}
	return b
}

func (b BgPageRes) ErrorDetail(err error) BgPageRes {
	if err != nil {
		b.Data = err.Error()
	}
	return b
}

func (b BgPageRes) ErrorDes(errDes string) BgPageRes {
	b.Data = errDes
	return b
}

func (b BgPageRes) Error() string {
	return fmt.Sprintf("code:%d Message:%s", b.Code, b.Msg)
}
