package service

import (
	"errors"
	"fmt"
	"github.com/ghodss/yaml"
	json "github.com/json-iterator/go"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/bean/entity"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/bean/vo/request"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/bean/vo/response"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/common/client"
	"gitlab.wodcloud.com/smart-operation/so-operation-api/src/pkg/beagle/resp"
	"os"
	"os/exec"
	"time"
)

type TaskManageSvc struct {
	User *entity.SystemUser
}

// AddTaskManage 新增任务
func (t *TaskManageSvc) AddTaskManage(req request.AddTaskManageReq) (err error) {
	db, err := client.GetDbClient()
	if err != nil {
		err = resp.DbConnectError.WithError(err)
		return
	}

	// 校验 任务名称 是否重复
	has, err := db.Table("task_manage").Where("is_delete = 0 AND task_name = ?", req.TaskName).Exist()
	if has {
		err = resp.DbDataCheckError.WithError(errors.New("任务名称重复"))
		return
	}
	if err != nil {
		err = resp.DbSelectError.WithError(err)
		return
	}

	taskManage := entity.TaskManage{
		TaskName:    req.TaskName,
		TaskDesc:    req.TaskDesc,
		YamlDesc:    req.YamlDesc,
		YamlUrl:     req.YamlUrl,
		HostGroupId: req.HostGroupId,
		CreateUser:  "",
		CreateTime:  time.Now(),
		UpdateUser:  "",
		UpdateTime:  time.Now(),
	}
	_, err = db.Table("task_manage").Insert(&taskManage)
	if err != nil {
		err = resp.DbInsertError.WithError(err)
		return
	}
	return
}

// EditTaskManage 编辑任务
func (t *TaskManageSvc) EditTaskManage(req request.EditTaskManageReq) (err error) {
	db, err := client.GetDbClient()
	if err != nil {
		err = resp.DbConnectError.WithError(err)
		return
	}

	taskManage := entity.TaskManage{
		TaskDesc:    req.TaskDesc,
		YamlDesc:    req.YamlDesc,
		YamlUrl:     req.YamlUrl,
		HostGroupId: req.HostGroupId,
		UpdateUser:  "",
		UpdateTime:  time.Now(),
	}
	//编辑任务
	_, err = db.Table("task_manage").Where("is_delete = 0 AND id = ?", req.Id).
		Cols("task_desc,yaml_desc,yaml_url,host_group_id,update_user,update_time").Update(&taskManage)
	if err != nil {
		err = resp.DbUpdateError.WithError(err)
		return
	}
	return
}

// DelTaskManage 删除任务
func (t *TaskManageSvc) DelTaskManage(req request.DelTaskManageReq) (err error) {
	db, err := client.GetDbClient()
	if err != nil {
		err = resp.DbConnectError.WithError(err)
		return
	}

	//编辑任务
	_, err = db.Table("task_manage").In("id", req.Id).Cols("is_delete").Update(&entity.TaskManage{
		IsDelete: 1,
	})
	if err != nil {
		err = resp.DbDeleteError.WithError(err)
		return
	}
	return
}

// DetailsTaskManage 任务详情
func (t *TaskManageSvc) DetailsTaskManage(id int) (taskManageRes response.TaskManageRes, err error) {
	db, err := client.GetDbClient()
	if err != nil {
		err = resp.DbConnectError.WithError(err)
		return
	}

	//查询任务详情
	finder := db.Table("task_manage").Alias("tm").
		Where("is_delete = 0 AND id = ?", id)
	_, err = finder.Select("tm.id,tm.task_name,tm.task_desc,tm.yaml_desc,tm.yaml_url,tm.create_user,tm.create_time," +
		"(select count(1) from task_history th where th.task_id = tm.id) as exec_cnt," +
		"(select count(1) from task_history th where th.task_id = tm.id and th.state = 1) as success_cnt," +
		"(select count(1) from task_history th where th.task_id = tm.id and th.state = 2) as fail_cnt").Get(&taskManageRes)
	if err != nil {
		err = resp.DbSelectError.WithError(err)
		return
	}

	//查询主机列表
	hostList := make([]response.HostList, 0)
	err = db.Table("host_manage_list").Where("is_delete = 0 AND host_group_id = ?", id).Find(&hostList)
	if err != nil {
		err = resp.DbSelectError.WithError(err)
		return
	}
	for _, v := range hostList {
		if v.HostFileUrl != "" {
			taskManageRes.HostFileUrl = v.HostFileUrl
		}

	}
	taskManageRes.HostList = hostList
	return
}

// ListTaskManage 任务列表
func (t *TaskManageSvc) ListTaskManage(req request.ListTaskManageReq) (total int64, taskManageListRes []response.TaskManageListRes, err error) {
	db, err := client.GetDbClient()
	if err != nil {
		err = resp.DbConnectError.WithError(err)
		return
	}

	finder := db.Table("task_manage").Alias("tm").Where("tm.is_delete = 0")
	if req.Search != "" {
		finder.Where(fmt.Sprintf("(tm.task_name LIKE %s OR tm.task_desc LIKE %s OR tm.create_user LIKE %s)",
			"%"+req.Search+"%", "%"+req.Search+"%", "%"+req.Search+"%"))
	}
	if req.CreateDateFrom != "" {
		finder.Where("tm.create_time >= ?", req.CreateDateFrom)
	}
	if req.CreateDateTo != "" {
		finder.Where("tm.create_time <= ?", req.CreateDateTo)
	}
	if req.HostGroupId != 0 {
		finder.Where("tm.host_group_id = ?", req.HostGroupId)
	}
	finder.OrderBy("tm.id")
	//查询任务
	total, err = finder.Select("tm.id,tm.task_name,tm.task_desc,(select count(*) from task_history th "+
		"where th.task_id = tm.id group by th.task_id) as exec_cnt,tm.create_user,tm.create_time").
		Limit(req.PageSize, (req.Page-1)*req.PageSize).FindAndCount(&taskManageListRes)
	if err != nil {
		err = resp.DbSelectError.WithError(err)
		return
	}
	return
}

func (t *TaskManageSvc) ExecScript(req request.ExecScriptReq) (data interface{}, err error) {
	var script map[string]interface{}
	j2, err := yaml.YAMLToJSON([]byte(req.Script))
	if err != nil {
		return nil, resp.YamlAnalysisError.WithError(err)
	}
	err = json.Unmarshal(j2, &script)
	if err != nil {
		return nil, resp.MarshalError.WithError(err)
	}
	if script["host"] == "all" {
		script["host"] = fmt.Sprintf("%s%d", AnsibleGroup, req.HostGroupId)
	}
	j, err := json.Marshal(script)
	if err != nil {
		return nil, resp.MarshalError.WithError(err)
	}
	y, err := yaml.JSONToYAML(j)
	if err != nil {
		return nil, resp.YamlAnalysisError.WithError(err)
	}
	req.Script = string(y)
	//写入执行脚本
	f2, err := os.Create("/etc/ansible/ansible.yml")
	if err != nil {
		return nil, resp.FileExecError.WithError(err)
	}
	defer f2.Close()
	_, err = f2.Write([]byte(req.Script))
	if err != nil {
		return nil, resp.FileExecError.WithError(err)
	}
	//写入额外yml参数
	if req.Type == 1 {
		//写入执行脚本
		f3, err := os.Create("/etc/ansible/ansible_extra.yml")
		if err != nil {
			return nil, resp.FileExecError.WithError(err)
		}
		defer f3.Close()
		_, err = f3.Write([]byte(req.Value))
		if err != nil {
			return nil, resp.FileExecError.WithError(err)
		}
		req.Value = fmt.Sprintf("@/etc/ansible/ansible_extra.yml")
	}
	var cmd *exec.Cmd
	if req.Value != "" {
		cmd = exec.Command("ansible-playbook", "-i", "/etc/ansible/hosts", "/etc/ansible/ansible.yml", "--extra-vars", req.Value)
	} else {
		cmd = exec.Command("ansible-playbook", "-i", "/etc/ansible/hosts", "/etc/ansible/ansible.yml")
	}
	output, err := cmd.Output()
	if err != nil {
		err = resp.CmdExecError.WithError(err)
		return
	}
	fmt.Println(string(output))
	return string(output), nil
}
