package com.pms.ocp.common.constants;


import com.pms.ocp.common.exception.ServiceException;
import org.apache.poi.ss.usermodel.Workbook;

import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URLEncoder;
import java.util.Locale;
import java.util.regex.Pattern;

public class ExportUtils {

    /**
     * 校验Header Manipulation
     * @param header 参数
     * @return 数据
     */
    public static String headerManipulation(String header) {
        if(StringUtils.isNullOrEmpty(header)){
            return header;
        }
        String regex = "[`~!@#$%^&*()\\+\\=\\{}|:\"?><【】\\/r\\/n]";
        Pattern pattern = Pattern.compile(regex);
        pattern.matcher(header).replaceAll("").trim();
        return header;
    }

    /**
     * 流导出文件至浏览器
     *
     * @param response
     * @param in
     * @param fileName
     */
    public static void exportWorld(HttpServletResponse response, InputStream in, String fileName) {
        OutputStream out = null;
        try {
            response.reset();
            String fileHeaderName = URLEncoder.encode(fileName, Constant.Character.UTF8);
            response.setHeader("Content-Disposition", "attachment;filename=" + headerManipulation(fileHeaderName));
            response.setContentType("application/msword;charset=UTF-8");
            response.setHeader("Access-Control-Expose-Headers", "Content-Disposition");
            response.setHeader("Access-Control-Allow-Origin", "*");
            out = response.getOutputStream();
            //定义缓存
            byte[] buffer = new byte[1024];
            int i;
            while ((i = in.read(buffer)) != Constant.Number.MINUS_ZERO) {
                out.write(buffer, Constant.Number.ZERO, i);
            }
            out.flush();
        } catch (IOException e) {
            e.printStackTrace();
            throw new ServiceException(com.pms.ocp.common.constants.ResultCode.INTERNAL_SERVER_ERROR);
        } finally {
            if (out != null){
                try {
                    out.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
                ;
            }
            if (in != null) {
                try {
                    in.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
    }

    /**
     * Workbook导出浏览器
     * 文件导出添加跨域 Access-Control-Allow-Origin
     * @param response
     * @param fileName
     * @param workbook
     */
    public static void exportExcel(HttpServletResponse response, String fileName, Workbook workbook) {
        try {
            response.reset();
            String fileHeaderName = URLEncoder.encode(fileName, Constant.Character.UTF8);
            response.setHeader("Content-Disposition", "attachment;filename=" + headerManipulation(fileHeaderName));
            response.setHeader("Access-Control-Expose-Headers", "Content-Disposition");
            response.setHeader("Access-Control-Allow-Origin", "*");
            response.setContentType("application/vnd.ms-excel;charset=UTF-8");
            ServletOutputStream outputStream = response.getOutputStream();
            workbook.write(outputStream);
        } catch (IOException e) {
            e.printStackTrace();
            throw new ServiceException(ResultCode.INTERNAL_SERVER_ERROR);
        }
    }

    /**
     * 图片导出
     *
     * @param response
     * @param in
     * @param fileName
     */
    public static void exportPic(HttpServletResponse response, InputStream in, String fileName) {
        try {
            if (in != null) {
                response.reset();
                String fileHeaderName = URLEncoder.encode(fileName, Constant.Character.UTF8);
                String fileExtensionName = fileName.substring(fileName.lastIndexOf(".") + 1).toUpperCase(Locale.ENGLISH);
                switch (fileExtensionName) {
                    case "JEPG":
                    case "JPG":
                        response.setContentType("image/jepg;charset=UTF-8");
                        break;
                    case "PNG":
                        response.setContentType("image/png;charset=UTF-8");
                        break;
                    case "BMP":
                        response.setContentType("application/x-bmp;charset=UTF-8");
                        break;
                    default:
                        response.setContentType("application/octet-stream;charset=UTF-8");
                        break;

                }
                response.setHeader("Content-Disposition", "attachment;filename=" + headerManipulation(fileHeaderName));
                response.setHeader("Access-Control-Expose-Headers", "Content-Disposition");
                response.setHeader("Access-Control-Allow-Origin", "*");
                byte[] buffer = new byte[1024];
                OutputStream out = response.getOutputStream();
                int i;
                while ((i = in.read(buffer)) != -1) {
                    out.write(buffer, 0, i);
                }
                out.flush();
                out.close();
                in.close();
            }
        } catch (IOException e) {
            e.printStackTrace();
            throw new ServiceException(ResultCode.INTERNAL_SERVER_ERROR);
        }
    }
}
