package com.pms.ocp.controller;

import com.github.pagehelper.Page;
import com.github.pagehelper.PageHelper;
import com.github.pagehelper.PageInfo;
import com.pms.ocp.common.utils.ExcelUtils;
import com.pms.ocp.model.dto.ModelDTO;
import com.pms.ocp.model.dto.ModelSubscribeDTO;
import com.pms.ocp.model.dto.ModelPropertyDTO;
import com.pms.ocp.model.dto.ModelRelationDTO;
import com.pms.ocp.model.entity.Model;
import com.pms.ocp.model.entity.ModelProperty;
import com.pms.ocp.model.entity.ModelSubscribe;
import com.pms.ocp.model.entity.ModelAudit;
import com.pms.ocp.model.vo.ModelAuditVO;
import com.pms.ocp.model.vo.ModelPropertyVO;
import com.pms.ocp.model.vo.ResponseVO;
import com.pms.ocp.service.*;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @Auther: wangjian
 * @Date: 2022/2/21 16:17
 * @Description:模型库管理接口
 */

@Slf4j
@RequestMapping("/model-base/v1")
@RestController
@Api(tags = "模型库管理接口")
public class ModelBaseController {

    @Autowired
    private ModelService modelService;

    @Autowired
    private ModelPropertyService modelPropertyService;

    @Autowired
    private ModelAuditService modelAuditService;

    @Autowired
    private ModelSubscribeService modelSubscribeService;

    @Autowired
    private ModelRelationService modelRelationService;

    /**
     * 模型-增加
     *
     * @return
     */
    @ApiOperation("模型-增加")
    @PostMapping("/create/model")
    public ResponseVO createModel(@RequestBody ModelDTO modelDTO) {

        return ResponseVO.ok(modelService.createModel(modelDTO));
    }

    /**
     * 模型-查询列表
     *
     * @return
     */
    @ApiOperation("模型-查询列表")
    @GetMapping("/get/model/list")
    public ResponseVO getModelList(@ApiParam(value = "模型订阅编号") @RequestParam(value = "objId", required = false) String objId,
                                   @ApiParam(value = "模型编码") @RequestParam(value = "modelCode", required = false) String modelCode,
                                   @ApiParam(value = "当前页") @RequestParam(value = "currentPage", required = false) Integer currentPage,
                                   @ApiParam(value = "每页数量") @RequestParam(value = "pageSize", required = false) Integer pageSize) {

        PageHelper.startPage(currentPage, pageSize);
        List<Model> modelList = modelService.getModelList(objId, modelCode);
        PageInfo<Model> pageInfo = new PageInfo<>(modelList);

        return ResponseVO.ok(pageInfo);
    }

    /**
     * 模型-删除
     *
     * @return
     */
    @ApiOperation("模型-删除")
    @PostMapping("/delete/model")
    public ResponseVO deleteModel(@ApiParam(value = "模型订阅编号") @RequestParam(value = "objId", required = false) String objId,
                                  @ApiParam(value = "模型编码") @RequestParam(value = "modelCode", required = false) String modelCode) {
        return ResponseVO.ok(modelService.deleteModel(objId, modelCode));
    }

    /**
     * 模型-修改
     *
     * @return
     */
    @ApiOperation("模型-修改")
    @PostMapping("/update/model")
    public ResponseVO updateModel(@RequestBody ModelDTO modelDTO) {
        return ResponseVO.ok(modelService.updateModel(modelDTO));
    }

    /**
     * 模型-导入
     *
     * @return
     */
    @ApiOperation("模型-导入")
    @PostMapping("/import/model")
    public ResponseVO importModel(@RequestParam("file") MultipartFile file) throws IOException {
        return ResponseVO.ok(modelService.importModel(file));
    }

    /**
     * 模型-下发/批量下发
     * TODO
     *
     * @return
     */
    @ApiOperation("模型-下发|批量下发")
    @PostMapping("/issue/model")
    public ResponseVO issueModel() {
        return ResponseVO.ok();
    }

    /**
     * 模型注册-上传
     *
     * @return
     */
    @ApiOperation("模型注册-上传")
    @PostMapping("/register/model")
    public ResponseVO registerModel() {
        return ResponseVO.ok();
    }

    /**
     * 模型订阅-增加
     *
     * @return
     */
    @ApiOperation("模型订阅-增加")
    @PostMapping("/create/model/subscribe")
    public ResponseVO createModelSubscribe(@RequestBody ModelSubscribeDTO modelSubscribeDTO) {
        return ResponseVO.ok(modelSubscribeService.createModelSubscribe(modelSubscribeDTO));
    }

    /**
     * 模型属性-删除/批量删除
     *
     * @return
     */
    @ApiOperation("模型订阅-删除/批量删除")
    @PostMapping("/delete/model/subscribe")
    public ResponseVO deleteModelSubscribe(@ApiParam(value = "订阅编号集合") @RequestBody List<String> ids) {
        return ResponseVO.ok(modelSubscribeService.deleteBatchModelSubscribe(ids));
    }

    /**
     * 模型订阅-查询
     *
     * @return
     */
    @ApiOperation("模型订阅-查询")
    @GetMapping("/get/model/subscribe")
    public ResponseVO getModelSubscribe(@ApiParam(value = "模型订阅编号") @RequestParam(value = "objId", required = false) String objId,
                                        @ApiParam(value = "模型编码") @RequestParam(value = "modelCode", required = false) String modelCode,
                                        @ApiParam(value = "订阅公司编码") @RequestParam(value = "subsCompanyCode", required = false) String subsCompanyCode,
                                        @ApiParam(value = "当前页") @RequestParam(value = "currentPage", required = false) Integer currentPage,
                                        @ApiParam(value = "每页数量") @RequestParam(value = "pageSize", required = false) Integer pageSize) {

        PageHelper.startPage(currentPage, pageSize);
        List<ModelSubscribe> modelSubscribeList = modelSubscribeService.getModelSubscribeList(objId, modelCode, subsCompanyCode);
        PageInfo<ModelSubscribe> pageInfo = new PageInfo<>(modelSubscribeList);

        return ResponseVO.ok(pageInfo);
    }

    /**
     * 模型事记-查询
     *
     * @return
     */
    @ApiOperation("模型事记-查询")
    @GetMapping("/get/model/note")
    public ResponseVO getModelNote(@RequestParam(value = "modelCode") String modelCode) {

        List<ModelAudit> modelAuditList = modelAuditService.getModelAuditList(modelCode);

        return ResponseVO.ok(modelAuditList);
    }

    /**
     * 模型事记-增加
     *
     * @return
     */
    @ApiOperation("模型事记-增加")
    @PostMapping("create/model/note")
    public ResponseVO createModelNote() {
        return ResponseVO.ok();
    }

    /**
     * 模型事记-导出
     *
     * @return
     */
    @ApiOperation("模型事记-导出")
    @PostMapping("export/model/note")
    public ResponseVO exportModelNote(HttpServletResponse response,
                                      @ApiParam(value = "模型编码") @RequestParam(value = "modelCode", required = false) String modelCode) throws IOException {
        // 查询要导出的数据
        List<ModelAudit> modelAuditList = modelAuditService.getModelAuditList(modelCode);

        //users集合转成export集合
        List<ModelAuditVO> exportVOList = modelAuditList.stream().map(modelNote -> {
            ModelAuditVO vo = new ModelAuditVO();
            BeanUtils.copyProperties(modelNote, vo);
            return vo;
        }).collect(Collectors.toList());

        ExcelUtils.exportExcel(response, "测试导出模型事记", "测试导出模型事记", "测试导出模型事记.xlsx", exportVOList, ModelAuditVO.class);

        return ResponseVO.ok();
    }

    /**
     * 模型属性-增加
     *
     * @return
     */
    @ApiOperation("模型属性-增加")
    @PostMapping("create/model/property")
    public ResponseVO createModelProperty(@RequestBody ModelPropertyDTO modelPropertyDTO) {

        Integer i = modelPropertyService.createModelProperty(modelPropertyDTO);
        return ResponseVO.ok(i);
    }

    /**
     * 模型属性-查询列表
     *
     * @return
     */
    @ApiOperation("模型属性-查询列表")
    @GetMapping("get/model/property/list")
    public ResponseVO getModelPropertyList(@ApiParam(value = "模型编码") @RequestParam(value = "modelCode", required = false) String modelCode,
                                           @ApiParam(value = "模型所属类型编码") @RequestParam(value = "modelGroupCode", required = false) String modelGroupCode,
                                           @ApiParam(value = "公司编码") @RequestParam(value = "ownerCompanyCode", required = false) String ownerCompanyCode,
                                           @ApiParam(value = "当前页") @RequestParam(value = "currentPage", required = false) Integer currentPage,
                                           @ApiParam(value = "每页数量") @RequestParam(value = "pageSize", required = false) Integer pageSize) {

//        PageHelper.startPage(currentPage, pageSize);
//        List<ModelProperty> modelPropertyVOList = modelPropertyService.listModelProperty(objId, modelCode, subsCompanyCode);
//        PageInfo<ModelProperty> pageInfo = new PageInfo<>(modelPropertyVOList);


        PageHelper.startPage(currentPage, pageSize);
        List<ModelPropertyVO> pageInfo = modelPropertyService.listModelPropertyVO(currentPage, pageSize, modelCode, ownerCompanyCode, modelGroupCode);
        PageInfo<ModelPropertyVO> pageInfo1 = new PageInfo<>(pageInfo);

        return ResponseVO.ok(pageInfo1);
    }

    /**
     * 模型属性-修改
     *
     * @return
     */
    @ApiOperation("模型属性-修改")
    @PostMapping("/update/model/property")
    public ResponseVO updateModelProperty(@RequestBody ModelPropertyDTO modelPropertyDTO) {

        return ResponseVO.ok(modelPropertyService.updateModelProperty(modelPropertyDTO));
    }

    /**
     * 模型属性-删除/批量删除
     *
     * @return
     */
    @ApiOperation("模型属性-删除/批量删除")
    @PostMapping("/delete/model/property")
    public ResponseVO deleteModelProperty(@ApiParam(value = "属性ID集合") @RequestBody List<String> ids) {
        return ResponseVO.ok(modelPropertyService.deleteBatchModelProperty(ids));
    }

    /**
     * 模型拓扑关系-增加
     *
     * @return
     */
    @ApiOperation("模型拓扑关系-增加")
    @PostMapping("create/model/topology")
    public ResponseVO createModelTopology(@RequestBody ModelRelationDTO modelRelationDTO) {

        Integer i = modelRelationService.createModelRelation(modelRelationDTO);
        return ResponseVO.ok(i);
    }

    /**
     * 模型拓扑关系-查询
     *
     * @return
     */
    @ApiOperation("模型拓扑关系-查询")
    @GetMapping("/get/model/topology")
    public ResponseVO getModelTopology(@ApiParam(value = "模型拓扑关系编号") @RequestParam(value = "objId", required = false) String objId,
                                       @ApiParam(value = "模型编码") @RequestParam(value = "modelCode", required = false) String modelCode) {
        return ResponseVO.ok(modelRelationService.getModelRelationList(objId, modelCode));
    }

    /**
     * 模型拓扑关系-删除
     *
     * @return
     */
    @ApiOperation("模型拓扑关系-删除")
    @PostMapping("/delete/model/topology")
    public ResponseVO deleteModelTopology(@ApiParam(value = "模型拓扑关系编号") @RequestParam(value = "objId", required = false) String objId,
                                          @ApiParam(value = "模型编码") @RequestParam(value = "modelCode", required = false) String modelCode) {
        return ResponseVO.ok(modelRelationService.deleteModelRelation(objId, modelCode));
    }


    @ApiOperation("模型列表-导航树")
    @GetMapping("/delete/model/modelListNavigation")
    public ResponseVO modelListNavigation(@RequestParam(required = false) String searchCondition) {
        return modelSubscribeService.modelListNavigation(searchCondition);
    }
}
