package com.pms.ocp.service.impl;

import cn.hutool.core.collection.CollectionUtil;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.pms.ocp.common.constants.CodeEnum;
import com.pms.ocp.common.utils.PageUtil;
import com.pms.ocp.mapper.ModelClassifyMapper;
import com.pms.ocp.mapper.OcpModelGroupMapper;
import com.pms.ocp.model.entity.ModelClassify;
import com.pms.ocp.model.vo.OcpModelGroupVO;
import com.pms.ocp.model.vo.ResponseVO;
import com.pms.ocp.service.ModelClassifyService;
import com.pms.ocp.service.OcpModelGroupService;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.*;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;

/**
 * @author huxiuwu
 * @version 1.0
 * @date 2022/3/8 15:20
 */
@Service
public class ModelClassifyServiceImpl implements ModelClassifyService {

    @Resource
    private ModelClassifyMapper classifyMapper;

    @Resource
    private OcpModelGroupService ocpService;

    @Resource
    private OcpModelGroupMapper ocpModelGroupMapper;

    /**
     * currentPage pageSize
     * 模型分类-导航树
     *
     * @return List<com.pms.ocp.model.vo.OcpModelGroupVO>
     * @author huxiuwu
     * @date 2022/3/8
     **/
    @Override
    public ResponseVO<List<OcpModelGroupVO>> getModelNavigation() {
        //根据是否传入搜索条件创建构造器进行查询
        List<OcpModelGroupVO> modelTypeList = this.selectByCondition(null, null);
        if (CollectionUtil.isEmpty(modelTypeList)) {
            return ResponseVO.error(CodeEnum.NO_DATA);
        }
        //对模型分类信息进行处理
        List<OcpModelGroupVO> resultList = this.build(modelTypeList);
        return ResponseVO.ok(resultList);
    }

    /**
     * 右侧子类详情列表
     * 这边是右侧详情页,需求是点击右侧导航树上的分类显示该分类的所有子集,
     * 所以第一次查询查询到所有并将其转换为key为其分类编码,value为其本身
     * 的map.第二次是根据搜素框的值进行模糊查询
     *
     * @return pms.ocp.model.vo.ResponseVO
     * @author huxiuwu
     * @date 2022/3/9 <PageInfo<OcpModelGroupVO>>
     **/
    @Override
    public ResponseVO subclassDetailsList(int currentPage,int pageSize,String parentGroupCode,String subclassGroupName) {
        //查询所有
        List<OcpModelGroupVO> ocpModelGroupVOList = this.selectByCondition(null,null);
        //将查询到所有数据的list转换为key为分类code,value为本身的map
        Map<String, OcpModelGroupVO> groupByCode = ocpModelGroupVOList.stream().collect(Collectors.toMap(OcpModelGroupVO::getModelGroupCode, Function.identity(), (key1, key2) -> key2));
        //根据key获取前端传入code的对象
        OcpModelGroupVO ocpModelGroupVO = groupByCode.get(parentGroupCode);
        //模糊查询
        List<OcpModelGroupVO> dimConditionList = this.selectByCondition(null,subclassGroupName);
        //递归获取模糊查询后属于parentGroupCode子集
        List<OcpModelGroupVO> childrenList = findChildren(dimConditionList, ocpModelGroupVO, new LinkedList<>());
        List<OcpModelGroupVO> sortList = childrenList.stream().sorted(Comparator.comparing(OcpModelGroupVO::getGroupCtime).reversed()).collect(Collectors.toList());
        PageUtil<OcpModelGroupVO> pageUtil = new PageUtil<>();
        pageUtil.startPage(currentPage, pageSize, sortList);
        return ResponseVO.ok(pageUtil);
    }

    /**
     * 修改
     *
     * @param list 修改list
     * @return pms.ocp.model.vo.ResponseVO
     * @author huxiuwu
     * @date 2022/3/9
     **/
    @Override
    public ResponseVO update(List<ModelClassify> list) {
        if (getModelClassify(list,()->new QueryWrapper<ModelClassify>().ne("obj_id",list.get(0).getObjId()))){
            return ResponseVO.error(CodeEnum.DATA_REPETITION);
        }
        boolean isUpdateSucceed = ocpService.saveOrUpdateBatch(list);
        if (!isUpdateSucceed) {
            ResponseVO.error(CodeEnum.UP_DATA_FAIL);
        }
        return ResponseVO.ok();
    }

    private boolean getModelClassify(List<ModelClassify> list,Supplier<QueryWrapper<ModelClassify>> supplier) {
        ModelClassify data = list.get(0);
        QueryWrapper<ModelClassify> queryWrapper = supplier.get();
        queryWrapper.and(QueryWrapper-> QueryWrapper.eq("model_group_code",data.getModelGroupCode()).or().eq("model_group_name",data.getModelGroupName()));
        List<ModelClassify> modelClassifyList = ocpModelGroupMapper.selectList(queryWrapper);
        if (CollectionUtil.isNotEmpty(modelClassifyList)){
            return true;
        }
        return false;
    }

    /**
     * 删除
     * @return pms.ocp.model.vo.ResponseVO
     * @author huxiuwu
     * @date 2022/3/9
     **/
    @Override
    public ResponseVO delete(String modelGroupCode) {
        if (!ocpService.removeById(modelGroupCode)) {
            return ResponseVO.error(CodeEnum.DELETE_FAIL);
        }
        return ResponseVO.ok();
    }

    /**
     * 新增
     *
     * @param list 新增分类列表
     * @return pms.ocp.model.vo.ResponseVO
     * @author huxiuwu
     * @date 2022/3/9
     **/
    @Override
    public ResponseVO insert(List<ModelClassify> list) {
        if (getModelClassify(list,()->new  QueryWrapper<ModelClassify>())){
            return ResponseVO.error(CodeEnum.DATA_REPETITION);
        }
        boolean result = ocpService.saveBatch(list);
        if (!result) {
            return ResponseVO.error(CodeEnum.INSERT_FAIL);
        }
        return ResponseVO.ok(list);
    }

    /**
     * 获取父节点下的所有子节点放入list中
     * @author huxiuwu
     * @date 2022/3/9
     * @param
     * @param
     * @param
     * @return void
     **/
    private List<OcpModelGroupVO> findChildren(List<OcpModelGroupVO> groupList, OcpModelGroupVO parent, List<OcpModelGroupVO> childList) {

        for (OcpModelGroupVO vo : groupList) {
            //遍历出父id等于参数的id，add进子节点集合
            if (parent != null
                && parent.getModelGroupCode()
                .equals(vo.getModelGroupPcode())
                && !"".equals(parent.getModelGroupCode())
                && !parent.getModelGroupCode().equals(vo.getModelGroupCode())) {
                parent.setHasChildren(true);
                vo.setModelGroupPName(parent.getModelGroupName());
                if (!childList.remove(vo)) {
                    childList.add(vo);
                }
                //递归下一级
                findChildren(groupList, vo, childList);
            }
        }
        return childList;
    }

    /**
     * 构建树节点
     *
     * @param treeNodes 需要构建的list
     * @return List<com.pms.ocp.model.vo.OcpModelGroupVO>
     * @author huxiuwu
     * @date 2022/3/10
     **/
    private List<OcpModelGroupVO> build(List<OcpModelGroupVO> treeNodes) {
        List<OcpModelGroupVO> result = new ArrayList<>();
        //list转map =>将每个对象的id作为key,自己作为value.
        Map<String, OcpModelGroupVO> nodeMap = new LinkedHashMap<>(treeNodes.size());
        for (OcpModelGroupVO treeNode : treeNodes) {
            nodeMap.put(treeNode.getModelGroupCode(), treeNode);
        }
        for (OcpModelGroupVO node : nodeMap.values()) {
            //得到自己的父类
            OcpModelGroupVO parent = nodeMap.get(node.getModelGroupPcode());
            if (parent != null && (node.getModelGroupPcode().equals(parent.getModelGroupCode()))
                    && (!"".equals(node.getModelGroupPcode())|| !"".equals(parent.getModelGroupCode()))
                    && !parent.getModelGroupCode().equals(node.getModelGroupCode())) {
                //防止每次都new list或者只是使用该对象就new对象浪费内存或者覆盖上一次记录
                if (parent.getChildrenList() == null) {
                    parent.setChildrenList(new LinkedList<>());
                }
                //设置上级父类名称
                node.setModelGroupPName(parent.getModelGroupName());
                //说明他有父类其下子类都需要加一级
                parent.getChildrenList().add(node);
                continue;
            }
            result.add(node);
        }
        return result;
    }

    /**
     * 生成构造器
     * @author huxiuwu
     * @date 2022/3/10
     * @param searchCondition
     * @return List<com.pms.ocp.model.vo.OcpModelGroupVO>
     * @author huxiuwu
     * @date 2022/3/10
     **/
    private List<OcpModelGroupVO> selectByCondition(String... searchCondition) {
        QueryWrapper<OcpModelGroupVO> queryWrapper = new QueryWrapper<>();
        //根据是否传入搜索条件创建构造器进行查询
        queryWrapper.eq(StringUtils.isNotEmpty(searchCondition[0]), "model_group_pcode", searchCondition[0])
        .like(StringUtils.isNotEmpty(searchCondition[1]),"model_group_name", searchCondition[1]);
        return classifyMapper.selectList(queryWrapper);
    }

    /**
     * 删除模型分类
     *
     * @param objId
     * @param modelGroupCode
     * @param groupCompanyCode
     * @return
     */
    @Override
    public Integer deleteModelClassify(String objId, String modelGroupCode, String groupCompanyCode) {

        QueryWrapper<OcpModelGroupVO> queryWrapper = new QueryWrapper();

        if (StringUtils.isNotEmpty(modelGroupCode)) {
            queryWrapper.lambda().eq(OcpModelGroupVO::getModelGroupCode, modelGroupCode);
        }

        List<OcpModelGroupVO> opList = classifyMapper.selectList(queryWrapper);

        if (!opList.isEmpty()) {
            if (StringUtils.isEmpty(objId)) {
                return 0;
            }
            if (StringUtils.isNotEmpty(objId)) {
                queryWrapper.lambda().eq(OcpModelGroupVO::getObjId, objId);
            }

            if (StringUtils.isNotEmpty(groupCompanyCode)) {
                queryWrapper.lambda().eq(OcpModelGroupVO::getGroupCompanyCode, groupCompanyCode);
            }
            return classifyMapper.delete(queryWrapper);
        }
        return -1;

    }
}
