package com.pms.ocp.service.impl;

import com.github.pagehelper.PageHelper;
import com.github.pagehelper.PageInfo;
import com.pms.ocp.common.constants.CodeEnum;
import com.pms.ocp.common.constants.Constant;
import com.pms.ocp.mapper.ModelCheckMapper;
import com.pms.ocp.model.dto.ModelAuditDto;
import com.pms.ocp.model.dto.OcpModelRequirementAcceptDto;
import com.pms.ocp.model.entity.ModelPropInfo;
import com.pms.ocp.model.vo.ModelCheckVo;
import com.pms.ocp.model.vo.ResponseVO;
import com.pms.ocp.service.ModelCheckService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author huxiuwu
 * @version 1.0
 * @date 2022/4/6 10:12
 */
@Slf4j
@Service
public class ModelCheckServiceImpl implements ModelCheckService {

    @Resource
    private ModelCheckMapper modelCheckMapper;

    /**
     * 模型审核列表
     * @author huxiuwu
     * @date 2022/4/6
     * @param currentPage 当前页
     * @param pageSize 每页数量
     * @param searchCondition
     * @return ResponseVO<java.util.List<com.pms.ocp.model.vo.ModelCheckVo>>
     **/
    @Override
    public ResponseVO<PageInfo<ModelCheckVo>> modelAuditList(Integer currentPage, Integer pageSize, String searchCondition) {
        PageHelper.startPage(currentPage,pageSize);
        List<ModelCheckVo> modelCheckVoList = modelCheckMapper.modelAuditList(searchCondition);
        //查询组织机构
        List<ModelCheckVo> companyList = modelCheckMapper.selectCompanyList();
        //将组织机构表合并到审核list中
        List<ModelCheckVo> result = pageList(modelCheckVoList,companyList);
        PageInfo<ModelCheckVo> pageInfo = new PageInfo<>(result);
        return ResponseVO.ok(pageInfo);
    }

    /**
     * 修改审核状态
     * @author huxiuwu
     * @date 2022/4/6
     * @param ocp
     * @return pms.ocp.model.vo.ResponseVO
     **/
    @Transactional(rollbackFor = Exception.class)
    @Override
    public ResponseVO updateAuditType(ModelAuditDto ocp) {
        //修改状态
        boolean isUpdateSucceed = modelCheckMapper.updateAuditState(ocp);
        boolean flag = isUpdateSucceed;
        if (ocp.getAuditState()== Constant.Number.ONE){
            OcpModelRequirementAcceptDto dto = pageOcpDto(ocp);
            //审核通过,新增至验收表
            boolean isInsertSucceed = modelCheckMapper.saveAccept(dto);
            flag = (isInsertSucceed && isUpdateSucceed) == true;
        }else{
            //其他状态删除该需求编码避免数据重复
            modelCheckMapper.deleteByRequirementCode(ocp);
        }
        if (!flag){
            ResponseVO.error(CodeEnum.DELETE_FAIL);
        }
        return ResponseVO.ok();
    }

    /**
     * 模型关系表
     * @author huxiuwu
     * @date 2022/4/7
     * @param modelCode
     * @return pms.ocp.model.vo.ResponseVO
     **/
    @Override
    public ResponseVO<List<ModelPropInfo>> modelRelation(String modelCode) {
        //查询模型以及自己所拥有字段
        List<ModelPropInfo> modelPropInfoList = modelCheckMapper.selectModelOwnerProp();
        //转换为map
        Map<String, ModelPropInfo> modelMap = modelPropInfoList.stream().collect(Collectors.toMap(ModelPropInfo::getModelCode, o -> o));
        //查询中间表获取关联模型code列表
        List<String> modelCodeList = modelCheckMapper.selectModelCodeList(modelCode);
        //取出map中值塞入实体类
        List<ModelPropInfo> list = pageModelRelationVo(modelCodeList,modelMap,modelCode);
        return ResponseVO.ok(list);
    }

    /**
     * 封装模型关联vo
     * @author huxiuwu
     * @date 2022/4/7
     * @param modelCodeList
     * @param modelMap
     * @param modelCode
     * @return pms.ocp.model.vo.ModelPropRelationVo
     **/
    private List<ModelPropInfo> pageModelRelationVo(List<String> modelCodeList, Map<String, ModelPropInfo> modelMap, String modelCode) {
        List<ModelPropInfo> list = new LinkedList<>();
        list.add(modelMap.get(modelCode));
        modelCodeList.forEach(v->
            list.add(modelMap.get(v))
        );
        return list;
    }

    /**
     * 封装审核通过新增验收表数据
     * @author huxiuwu
     * @date 2022/4/6
     * @param ocp
     * @return pms.ocp.model.dto.OcpModelRequirementAcceptDto
     **/
    private OcpModelRequirementAcceptDto pageOcpDto(ModelAuditDto ocp) {
        OcpModelRequirementAcceptDto dto = new OcpModelRequirementAcceptDto();
        dto.setObjId(UUID.randomUUID().toString());
        dto.setAcceptCompanyCode("");
        dto.setAcceptDepartment("");
        dto.setAcceptUserName("");
        dto.setIsDelete(0);
        dto.setAcceptCtime(new Date());
        dto.setAcceptMtime(new Date());
        dto.setAcceptStats(0);
        dto.setRequirementCode(ocp.getRequirementCode());
        dto.setAcceptUserName("");
        dto.setAcceptUserId("");
        return dto;
    }

    /**
     *
     * @author huxiuwu
     * @date 2022/4/6
     * @param modelCheckVoList
     * @param companyList
     * @return List<com.pms.ocp.model.vo.ModelCheckVo>
     **/
    private List<ModelCheckVo> pageList(List<ModelCheckVo> modelCheckVoList, List<ModelCheckVo> companyList) {
        List<ModelCheckVo> list = modelCheckVoList.stream().map(m -> {
            companyList.stream().filter(m2-> Objects.equals(m.getProposerCompanyCode(),m2.getProposerCompanyCode())).forEach(m2-> {
                m.setProposerCompanyName(m2.getProposerCompanyName());
            });
            return m;
        }).collect(Collectors.toList());
        return list;
    }
}
