package com.pms.ocp.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.pms.ocp.common.constants.Constant;
import com.pms.ocp.common.utils.ExcelUtils;
import com.pms.ocp.common.utils.PageUtil;
import com.pms.ocp.mapper.ModelMapper;
import com.pms.ocp.mapper.ModelPropertyMapper;
import com.pms.ocp.model.dto.ModelDTO;
import com.pms.ocp.model.entity.Model;
import com.pms.ocp.model.entity.ModelProperty;
import com.pms.ocp.model.vo.ModelVO;
import com.pms.ocp.model.vo.ResponseVO;
import com.pms.ocp.service.ModelService;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;

import javax.annotation.Resource;
import java.io.IOException;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;

/**
 * @Auther: wangjian
 * @Date: 2022/2/22 16:21
 * @Description:模型业务层实现
 */

@Service
public class ModelServiceImpl extends ServiceImpl<ModelMapper, Model> implements ModelService {

    @Resource
    private ModelMapper modelMapper;

    @Resource
    private ModelPropertyMapper modelPropertyMapper;

    /**
     * 创建模型
     *
     * @param modelDTO
     * @return
     */
    @Override
    public Integer createModel(ModelDTO modelDTO) {

        Model model = new Model();

        BeanUtils.copyProperties(modelDTO, model);

        model.setModelCtime(LocalDateTime.now());
        int result = modelMapper.insert(model);
        return result;
    }

    /**
     * 删除模型
     *
     * @param objId
     * @return
     */
    @Override
    public Integer deleteModel(String objId, String modelCode) {
        QueryWrapper<Model> queryWrapper = new QueryWrapper();

        if (StringUtils.isNotEmpty(objId)) {
            queryWrapper.lambda().eq(Model::getObjId, objId);
        }

        if (StringUtils.isNotEmpty(modelCode)) {
            queryWrapper.lambda().eq(Model::getModelCode, modelCode);
        }

        return modelMapper.delete(queryWrapper);

    }

    /**
     * 更新模型
     *
     * @param modelDTO
     * @return
     */
    @Override
    public Integer updateModel(ModelDTO modelDTO) {
        Model model = new Model();

        BeanUtils.copyProperties(modelDTO, model);

        model.setModelMtime(LocalDateTime.now());

        UpdateWrapper<Model> updateWrapper = new UpdateWrapper<>();

        updateWrapper.lambda().eq(Model::getObjId, model.getObjId());

        return modelMapper.update(model, updateWrapper);
    }

    /**
     * 导入模型
     *
     * @param file
     * @return
     */
    @Override
    public Boolean importModel(MultipartFile file) throws IOException {

        //校验Excel文件
        ExcelUtils.checkFile(file);
        List<ModelVO> modelVOList = ExcelUtils.importExcel(file, ModelVO.class);
        List<Model> modelList = new ArrayList<>(modelVOList.size());

        if (!modelVOList.isEmpty()) {
            for (ModelVO modelVO : modelVOList) {
                Model model = new Model();
                BeanUtils.copyProperties(modelVO, model);
                model.setModelCtime(LocalDateTime.now());
                modelList.add(model);
            }
        }

        //校验Excel数据
//        checkData(excelCps);
        return modelMapper.insertBatchModel(modelList);

    }

    /**
     * 获取模型列表
     *
     * @param
     * @return
     */
    @Override
    public List<Model> getModelList(String objId, String modelCode, List<String> ownerCompanyCodes, List<String> modelGroupCodes) {
        QueryWrapper<Model> queryWrapper = new QueryWrapper();

        if (StringUtils.isNotEmpty(objId)) {
            queryWrapper.lambda().eq(Model::getObjId, objId);
        }

        if (StringUtils.isNotEmpty(modelCode)) {
            queryWrapper.lambda().eq(Model::getModelCode, modelCode);
        }

        if (ownerCompanyCodes != null && !ownerCompanyCodes.isEmpty()) {
            List<Model> modelList = new ArrayList<>();
            for (String ownerCompanyCode : ownerCompanyCodes) {
                queryWrapper.lambda().eq(Model::getOwnerCompanyCode, ownerCompanyCode);
                List<Model> modelList1;
                modelList1 = modelMapper.selectList(queryWrapper);
                modelList.addAll(modelList1);
                queryWrapper.lambda().clear();
            }
            return modelList;
        }

        if (modelGroupCodes != null && !modelGroupCodes.isEmpty()) {
            List<Model> modelList = new ArrayList<>();
            for (String modelGroupCode : modelGroupCodes) {
                queryWrapper.lambda().eq(Model::getModelGroupCode, modelGroupCode);
                List<Model> modelList1;
                modelList1 = modelMapper.selectList(queryWrapper);
                modelList.addAll(modelList1);
                queryWrapper.lambda().clear();
            }
            return modelList;
        }

        if (StringUtils.isNotEmpty(modelCode)) {
            queryWrapper.lambda().eq(Model::getModelCode, modelCode);
        }

        return modelMapper.selectList(queryWrapper);
    }

    /**
     * 获取模型
     *
     * @param objId
     * @return
     */
    @Override
    public Model getModel(String objId, String modelCode) {
        QueryWrapper<Model> queryWrapper = new QueryWrapper<>();

        if (StringUtils.isNotEmpty(objId)) {
            queryWrapper.lambda().eq(Model::getObjId, objId);
        }

        if (StringUtils.isNotEmpty(modelCode)) {
            queryWrapper.lambda().eq(Model::getModelCode, modelCode);
        }

        return modelMapper.selectOne(queryWrapper);
    }

    @Override
    public ResponseVO getModeBaselList(String belongLevel, String modelCode, Integer currentPage, Integer pageSize, String searchCondition) {
        ResponseVO result;
        switch (belongLevel) {
            case Constant.Character.STRINGTWO:
                result = this.getModelInfoList(modelCode, currentPage, pageSize, searchCondition);
                break;
            case Constant.Character.STRINGTHREE:
                result = this.getPropertyList(modelCode, currentPage, pageSize, searchCondition);
                break;
            default:
                return ResponseVO.ok();
        }
        return result;
    }

    /**
     * @param modelCode       分类编码
     * @param currentPage     当前页
     * @param pageSize        每页数量
     * @param searchCondition 搜索条件
     * @return pms.ocp.model.vo.ResponseVO
     * @author huxiuwu
     * @date 2022/3/17
     **/
    private ResponseVO getModelInfoList(String modelCode, Integer currentPage, Integer pageSize, String searchCondition) {
        QueryWrapper<Model> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("model_group_code", modelCode)
        .like(StringUtils.isNotEmpty(searchCondition), "model_name", searchCondition)
        .orderByDesc("model_ctime");
        List<Model> modelList = modelMapper.selectList(queryWrapper);
        PageUtil<Model> pageUtil = new PageUtil<>();
        pageUtil.startPage(currentPage, pageSize, modelList);
        return ResponseVO.ok(pageUtil);
    }

    /**
     * @param modelCode       分类编码
     * @param currentPage     当前页
     * @param pageSize        每页数量
     * @param searchCondition 搜索条件
     * @return pms.ocp.model.vo.ResponseVO
     * @author huxiuwu
     * @date 2022/3/17
     **/
    private ResponseVO getPropertyList(String modelCode, Integer currentPage, Integer pageSize, String searchCondition) {
        QueryWrapper<ModelProperty> proWrapper = new QueryWrapper<>();
        //查询模型名称
        QueryWrapper<Model> modelWrapper = new QueryWrapper<>();
        modelWrapper.eq("model_code",modelCode);
        List<Model> modelList = modelMapper.selectList(modelWrapper);
        proWrapper.eq("model_code", modelCode)
        .like(StringUtils.isNotEmpty(searchCondition), "column_comments", searchCondition)
        .orderByDesc("prop_ctime");
        List<ModelProperty> modelPropertiesList = modelPropertyMapper.selectList(proWrapper);
        modelPropertiesList.forEach(pro ->
            pro.setModelTable(modelList.get(0) == null ? "":modelList.get(0).getModelTable())
        );
        PageUtil<ModelProperty> pageUtil = new PageUtil<>();
        pageUtil.startPage(currentPage, pageSize, modelPropertiesList);
        return ResponseVO.ok(pageUtil);
    }
}
