package com.pms.ocp.common.aspectj;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.pms.ocp.common.utils.StringBusinessUtil;
import lombok.extern.slf4j.Slf4j;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.*;
import org.springframework.context.annotation.Profile;
import org.springframework.stereotype.Component;
import org.springframework.validation.BindingResult;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.Map;

@Aspect
@Component
@Slf4j
/**
 * 根据配置可在不同环境中生效
 * @Profile({"dev", "test"})
 */
//@Profile({"dev", "test"})
public class WebLogAspect {

    private static final String LINE_SEPARATOR = System.lineSeparator();

    @Pointcut("@annotation(com.pms.ocp.common.aspectj.WebLog)")
    public void webLog() {
    }

    /**
     * 环绕
     *
     * @param proceedingJoinPoint
     * @return
     * @throws Throwable
     */
    @Around("webLog()")
    public Object doAround(ProceedingJoinPoint proceedingJoinPoint) throws Throwable {
        long startTime = System.currentTimeMillis();
        Object result = proceedingJoinPoint.proceed();
        // 打印出参
        log.info("Response Args  : {}", JSONObject.toJSONString(result));
        // 打印耗时
        log.info("Time-Consuming : {} ms", System.currentTimeMillis() - startTime);
        return result;
    }

    /**
     * 在切点前切入
     *
     * @param joinPoint
     * @throws Throwable
     */
    @Before("webLog()")
    public void doBefore(JoinPoint joinPoint) throws Throwable {
        ServletRequestAttributes attributes = (ServletRequestAttributes) RequestContextHolder.getRequestAttributes();
        HttpServletRequest request = attributes.getRequest();
        // 获取 @WebLog 注解的描述信息
        String methodDescription = getAspectLogDescription(joinPoint);
        // 处理入参
        Object[] args = joinPoint.getArgs();
        Map<String, Object> map = new HashMap<>();
        for (Object arg : args) {
            if (StringBusinessUtil.isNullOrEmpty(arg)) {
                continue;
            }
            if (arg instanceof BindingResult) {
                continue;
            }
            if (arg instanceof HttpSession) {
                continue;
            }
            if (arg instanceof HttpServletRequest) {
                continue;
            }
            if (arg instanceof HttpServletResponse) {
                continue;
            }
            if (arg instanceof MultipartFile) {
                MultipartFile file = (MultipartFile) arg;
                map.put("file", getFileParam(file));
            } else if (arg instanceof MultipartFile[]) {
                MultipartFile[] files = (MultipartFile[]) arg;
                Map<String, Object> t = new HashMap<>();
                for (MultipartFile file : files) {
                    t.put("file", getFileParam(file));
                }
                map.put("files", t);
            } else {
                try {
                    map.put(arg.getClass().getSimpleName(), arg);
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        }
        // 打印请求相关参数
        log.info("============================================ 接口开始执行 ==================================================");
        try {
            // 打印请求 url
            log.info("URL           : {}", request.getRequestURL().toString());
            // 打印描述信息
            log.info("Description   : {}", methodDescription);
            // 打印 Http method
            log.info("HTTP Method   : {}", request.getMethod());
            // 打印调用 controller 的全路径以及执行方法
            log.info("Class Method  : {}.{}", joinPoint.getSignature().getDeclaringTypeName(), joinPoint.getSignature().getName());
            // 打印请求的IP
            log.info("IP            : {}", request.getRemoteAddr());
            //log.info("Request Args  : {}", new Gson().toJson(joinPoint.getArgs()));
            // 打印请求的入参
            log.info("Request Args  : {}", JSON.toJSONString(map));
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * 处理文件，获取文件基本信息
     *
     * @param file
     * @return
     */
    private Map<String, Object> getFileParam(MultipartFile file) {
        Map<String, Object> map = null;
        try {
            map = new HashMap<>();
            map.put("文件名:", file.getOriginalFilename());
            map.put("文件类型:", file.getContentType());
            map.put("文件大小:", file.getSize());
        } catch (Exception e) {
            e.printStackTrace();
        }
        return map;
    }

    /**
     * 在切点后切入
     *
     * @throws Throwable
     */
    @After("webLog()")
    public void doAfter() throws Throwable {
        log.info("============================================ 接口执行结束 ===================================================" + LINE_SEPARATOR);
    }

    /**
     * 获取切面注解的描述
     */
    private String getAspectLogDescription(JoinPoint joinPoint) throws ClassNotFoundException {
        String targetName = joinPoint.getTarget().getClass().getName();
        String methodName = joinPoint.getSignature().getName();
        Object[] arguments = joinPoint.getArgs();
        Class targetClass = Class.forName(targetName);
        Method[] methods = targetClass.getMethods();
        StringBuilder description = new StringBuilder("");
        for (Method method : methods) {
            if (method.getName().equals(methodName)) {
                Class[] calzzs = method.getParameterTypes();
                if (calzzs.length == arguments.length) {
                    description.append(method.getAnnotation(WebLog.class).description());
                    break;
                }
            }
        }
        return description.toString();
    }
}
