package org.dromara.zjk.learning_world.controller;

import cn.dev33.satoken.annotation.SaCheckRole;
import cn.dev33.satoken.annotation.SaMode;
import jakarta.validation.constraints.NotEmpty;
import jakarta.validation.constraints.NotNull;
import lombok.RequiredArgsConstructor;
import org.dromara.common.core.constant.TenantConstants;
import org.dromara.common.core.domain.R;
import org.dromara.common.core.validate.AddGroup;
import org.dromara.common.core.validate.EditGroup;
import org.dromara.common.idempotent.annotation.RepeatSubmit;
import org.dromara.common.log.annotation.Log;
import org.dromara.common.log.enums.BusinessType;
import org.dromara.common.mybatis.core.page.PageQuery;
import org.dromara.common.mybatis.core.page.TableDataInfo;
import org.dromara.common.web.core.BaseController;
import org.dromara.zjk.learning_world.domain.bo.ZjkLearningInformationBo;
import org.dromara.zjk.learning_world.domain.vo.ZjkLearningInformationVo;
import org.dromara.zjk.learning_world.service.IZjkLearningInformationService;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * 资讯
 * 前端访问路由地址为:/zjk/learningInformation
 *
 * @author ygh
 * @date 2024-11-11
 */
@Validated
@RequiredArgsConstructor
@RestController
@RequestMapping("/learningInformation")
public class ZjkLearningInformationController extends BaseController {

    private final IZjkLearningInformationService zjkLearningInformationService;

    /**
     * 查询资讯列表
     */
    @SaCheckRole(value={
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    },mode = SaMode.OR)
    @GetMapping("/list")
    public TableDataInfo<ZjkLearningInformationVo> list(ZjkLearningInformationBo bo, PageQuery pageQuery) {
        return zjkLearningInformationService.queryPageList(bo, pageQuery);
    }


    /**
     * 获取资讯详细信息
     *
     * @param id 主键
     */
    @SaCheckRole(value={
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    },mode = SaMode.OR)
    @GetMapping("/{id}")
    public R<ZjkLearningInformationVo> getInfo(@NotNull(message = "主键不能为空")
                                     @PathVariable Long id) {
        return R.ok(zjkLearningInformationService.queryById(id));
    }

    /**
     * 新增资讯
     */
    @SaCheckRole(value={
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,

    },mode = SaMode.OR)
    @Log(title = "资讯", businessType = BusinessType.INSERT)
    @RepeatSubmit()
    @PostMapping()
    public R<Void> add(@Validated(AddGroup.class) @RequestBody ZjkLearningInformationBo bo) {
        return toAjax(zjkLearningInformationService.insertByBo(bo));
    }

    /**
     * 修改资讯
     */
    @SaCheckRole(value={
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "资讯", businessType = BusinessType.UPDATE)
    @RepeatSubmit()
    @PostMapping("/update")
    public R<Void> edit(@Validated(EditGroup.class) @RequestBody ZjkLearningInformationBo bo) {
        return toAjax(zjkLearningInformationService.updateByBo(bo));
    }
    /**
     * 修改资讯 状态
     */
    @SaCheckRole(value={
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "资讯", businessType = BusinessType.UPDATE)
    @RepeatSubmit()
    @PostMapping("/update/status")
    public R<Void> editStatus(@Validated(EditGroup.class) @RequestBody ZjkLearningInformationBo bo) {
        return toAjax(zjkLearningInformationService.updateStatus(bo));
    }

    /**
     * 删除资讯
     *
     * @param ids 主键串
     */
    @SaCheckRole(value={
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "资讯", businessType = BusinessType.DELETE)
    @GetMapping("/deleted/{ids}")
    public R<Void> remove(@NotEmpty(message = "主键不能为空")
                          @PathVariable Long[] ids) {
        return toAjax(zjkLearningInformationService.deleteWithValidByIds(List.of(ids), true));
    }

    /**
     * 获取帖子数量
     * @return
     */
    @SaCheckRole(value={
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    },mode = SaMode.OR)
    @GetMapping("/getCount")
    public R<Long> getCouont() {
        return R.ok("success",zjkLearningInformationService.getCount());
    }
}
