package org.dromara.zjk.statement.controller;

import cn.dev33.satoken.annotation.SaCheckPermission;
import cn.dev33.satoken.annotation.SaCheckRole;
import cn.dev33.satoken.annotation.SaMode;
import jakarta.servlet.http.HttpServletResponse;
import jakarta.validation.constraints.NotEmpty;
import jakarta.validation.constraints.NotNull;
import lombok.RequiredArgsConstructor;
import org.dromara.common.core.constant.TenantConstants;
import org.dromara.common.core.domain.R;
import org.dromara.common.core.validate.EditGroup;
import org.dromara.common.excel.utils.ExcelUtil;
import org.dromara.common.idempotent.annotation.RepeatSubmit;
import org.dromara.common.log.annotation.Log;
import org.dromara.common.log.enums.BusinessType;
import org.dromara.common.mybatis.core.page.PageQuery;
import org.dromara.common.mybatis.core.page.TableDataInfo;
import org.dromara.common.web.core.BaseController;
import org.dromara.zjk.statement.domain.bo.ZjkMoneyStatementBo;
import org.dromara.zjk.statement.domain.vo.ZjkMoneyStatementVo;
import org.dromara.zjk.statement.service.IZjkMoneyStatementService;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * 费用报表
 * 前端访问路由地址为:/zjk/moneyStatement
 *
 * @author ygh
 * @date 2024-12-16
 */
@Validated
@RequiredArgsConstructor
@RestController
@RequestMapping("/moneyStatement")
public class ZjkMoneyStatementController extends BaseController {

    private final IZjkMoneyStatementService zjkMoneyStatementService;

    /**
     * 查询费用报表列表
     */
    @SaCheckRole(value = {
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    },mode = SaMode.OR)
    @GetMapping("/list")
    public TableDataInfo<ZjkMoneyStatementVo> list(ZjkMoneyStatementBo bo, PageQuery pageQuery) {
        return zjkMoneyStatementService.queryPageList(bo, pageQuery);
    }

    /**
     * 导出费用报表列表
     */
    @SaCheckPermission("zjk:moneyStatement:export")
    @SaCheckRole(value = {
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "费用报表", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public void export(ZjkMoneyStatementBo bo, HttpServletResponse response) {
        List<ZjkMoneyStatementVo> list = zjkMoneyStatementService.queryList(bo);
        ExcelUtil.exportExcel(list, "费用报表", ZjkMoneyStatementVo.class, response);
    }

    /**
     * 获取费用报表详细信息
     *
     * @param id 主键
     */
    @SaCheckPermission("zjk:moneyStatement:query")
    @SaCheckRole(value = {
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    },mode = SaMode.OR)
    @GetMapping("/{id}")
    public R<ZjkMoneyStatementVo> getInfo(@NotNull(message = "主键不能为空")
                                     @PathVariable Long id) {
        return R.ok(zjkMoneyStatementService.queryById(id));
    }

    /**
     * 新增费用报表
     */
    @SaCheckRole(value = {
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "费用报表", businessType = BusinessType.INSERT)
    @RepeatSubmit()
    @PostMapping("/add")
    public R<Void> add() {
        return toAjax(zjkMoneyStatementService.insertByBo());
    }

    @SaCheckRole(value = {
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "费用报表", businessType = BusinessType.INSERT)
    @RepeatSubmit()
    @PostMapping("/add/year")
    public R<Void> addYear() {
        return toAjax(zjkMoneyStatementService.addYear());
    }
    /**
     * 修改费用报表
     */
    @SaCheckPermission("zjk:moneyStatement:edit")
    @SaCheckRole(value = {
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "费用报表", businessType = BusinessType.UPDATE)
    @RepeatSubmit()
    @PostMapping("/edit")
    public R<Void> edit(@Validated(EditGroup.class) @RequestBody ZjkMoneyStatementBo bo) {
        return toAjax(zjkMoneyStatementService.updateByBo(bo));
    }

    /**
     * 删除费用报表
     *
     * @param ids 主键串
     */
    @SaCheckPermission("zjk:moneyStatement:remove")
    @SaCheckRole(value = {
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "费用报表", businessType = BusinessType.DELETE)
    @GetMapping("/delete/{ids}")
    public R<Void> remove(@NotEmpty(message = "主键不能为空")
                          @PathVariable Long[] ids) {
        return toAjax(zjkMoneyStatementService.deleteWithValidByIds(List.of(ids), true));
    }
}
