以下是 Docker 镜像制作的核心命令详解及操作流程：

---

### 一、镜像构建核心命令
#### 1. **`docker build`** - 构建镜像
```bash
# 基础构建（使用当前目录的 Dockerfile）
docker build -t myapp:1.0 .

# 指定 Dockerfile 路径
docker build -f ./docker/Dockerfile.prod -t myapp:prod .

# 构建参数传递（Dockerfile 中使用 ARG 接收）
docker build --build-arg APP_VERSION=1.2 -t myapp:1.2 .
```


#### 2. **`docker image`** - 镜像管理
```bash
# 查看镜像列表
docker image ls

# 删除镜像（强制删除使用 -f）
docker image rm myapp:1.0

# 导出镜像到文件
docker save -o myapp.tar myapp:1.0

# 从文件导入镜像
docker load -i myapp.tar
```


---

### 二、镜像标记与推送
#### 1. **`docker tag`** - 打标签
```bash
# 为镜像添加新标签（常用于版本管理）
docker tag myapp:1.0 registry.example.com/myteam/myapp:latest
```


#### 2. **`docker push`** - 推送镜像
```bash
# 登录镜像仓库
docker login registry.example.com

# 推送镜像到仓库
docker push registry.example.com/myteam/myapp:latest
```


---

### 三、构建优化命令
#### 1. **多阶段构建**
```dockerfile
# 阶段1：构建环境
FROM maven:3.8 AS builder
COPY . .
RUN mvn package

# 阶段2：运行环境
FROM openjdk:11-jre-slim
COPY --from=builder /target/app.jar /app.jar
CMD ["java", "-jar", "app.jar"]
```


#### 2. **构建缓存控制**
```bash
# 禁用缓存构建
docker build --no-cache -t myapp:nocache .

# 指定缓存来源
docker build --cache-from=myapp:1.0 -t myapp:1.1 .
```


---

### 四、镜像分析命令
```bash
# 查看镜像构建历史（含层信息）
docker history myapp:1.0

# 分析镜像内容（文件/目录大小）
docker run --rm -it myapp:1.0 du -sh /*

# 安全扫描（需安装 docker-slim）
docker-slim build --target myapp:1.0 --http-probe
```


---

### 五、典型工作流示例
```bash
# 1. 构建镜像
docker build -t myapp:1.0 .

# 2. 本地测试运行
docker run -p 8080:8080 --rm myapp:1.0

# 3. 打标签并推送
docker tag myapp:1.0 myregistry.com/project/myapp:1.0
docker push myregistry.com/project/myapp:1.0

# 4. 清理旧镜像
docker image prune -f  # 删除悬空镜像
docker image rm myapp:0.9  # 删除特定版本
```


> 💡 **最佳实践建议**：
> 1. 使用 `.dockerignore` 文件排除构建无关文件（如 `node_modules`, `.git`）
> 2. 镜像标签遵循语义化版本（如 `myapp:1.2.3`）
> 3. 生产环境使用特定基础镜像（如 `distroless` 或 `alpine`）
> 4. 定期扫描镜像漏洞：`docker scan myapp:1.0`