package org.dromara.zjk.expert.expertPaper.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import lombok.RequiredArgsConstructor;
import org.dromara.common.core.utils.MapstructUtils;
import org.dromara.common.core.utils.StringUtils;
import org.dromara.common.mybatis.core.page.PageQuery;
import org.dromara.common.mybatis.core.page.TableDataInfo;
import org.dromara.zjk.expert.expertPaper.domain.ZjkExpertPaper;
import org.dromara.zjk.expert.expertPaper.domain.bo.ZjkExpertPaperBo;
import org.dromara.zjk.expert.expertPaper.domain.vo.ZjkExpertPaperVo;
import org.dromara.zjk.expert.expertPaper.mapper.ZjkExpertPaperMapper;
import org.dromara.zjk.expert.expertPaper.service.IZjkExpertPaperService;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;

/**
 * 专家论文信息Service业务层处理
 *
 * @author zhangzhou
 * @date 2024-11-10
 */
@RequiredArgsConstructor
@Service
public class ZjkExpertPaperServiceImpl implements IZjkExpertPaperService {

    private final ZjkExpertPaperMapper baseMapper;

    /**
     * 查询专家论文信息
     *
     * @param id 主键
     * @return 专家论文信息
     */
    @Override
    public ZjkExpertPaperVo queryById(Long id){
        return baseMapper.selectVoById(id);
    }

    /**
     * 分页查询专家论文信息列表
     *
     * @param bo        查询条件
     * @param pageQuery 分页参数
     * @return 专家论文信息分页列表
     */
    @Override
    public TableDataInfo<ZjkExpertPaperVo> queryPageList(ZjkExpertPaperBo bo, PageQuery pageQuery) {
        Page<ZjkExpertPaperVo> result = baseMapper.selectVoListJoinExpertId(pageQuery.build(), bo);
        return TableDataInfo.build(result);
    }

    /**
     * 查询符合条件的专家论文信息列表
     *
     * @param bo 查询条件
     * @return 专家论文信息列表
     */
    @Override
    public List<ZjkExpertPaperVo> queryList(ZjkExpertPaperBo bo) {
        LambdaQueryWrapper<ZjkExpertPaper> lqw = buildQueryWrapper(bo);
        return baseMapper.selectVoList(lqw);
    }

    private LambdaQueryWrapper<ZjkExpertPaper> buildQueryWrapper(ZjkExpertPaperBo bo) {
        Map<String, Object> params = bo.getParams();
        LambdaQueryWrapper<ZjkExpertPaper> lqw = Wrappers.lambdaQuery();
        lqw.eq(bo.getExpertId() != null, ZjkExpertPaper::getExpertId, bo.getExpertId());
        lqw.like(StringUtils.isNotBlank(bo.getPaperName()), ZjkExpertPaper::getPaperName, bo.getPaperName());
        lqw.eq(StringUtils.isNotBlank(bo.getPeriodicalNumber()), ZjkExpertPaper::getPeriodicalNumber, bo.getPeriodicalNumber());
        lqw.eq(StringUtils.isNotBlank(bo.getPaperPath()), ZjkExpertPaper::getPaperPath, bo.getPaperPath());
        return lqw;
    }

    /**
     * 新增专家论文信息
     *
     * @param bo 专家论文信息
     * @return 是否新增成功
     */
    @Override
    public Boolean insertByBo(ZjkExpertPaperBo bo) {
        ZjkExpertPaper add = MapstructUtils.convert(bo, ZjkExpertPaper.class);
        validEntityBeforeSave(add);
        boolean flag = baseMapper.insert(add) > 0;
        if (flag) {
            bo.setId(add.getId());
        }
        return flag;
    }

    /**
     * 修改专家论文信息
     *
     * @param bo 专家论文信息
     * @return 是否修改成功
     */
    @Override
    public Boolean updateByBo(ZjkExpertPaperBo bo) {
        ZjkExpertPaper update = MapstructUtils.convert(bo, ZjkExpertPaper.class);
        validEntityBeforeSave(update);
        return baseMapper.updateById(update) > 0;
    }

    /**
     * 保存前的数据校验
     */
    private void validEntityBeforeSave(ZjkExpertPaper entity){
        //TODO 做一些数据校验,如唯一约束
    }

    /**
     * 校验并批量删除专家论文信息信息
     *
     * @param ids     待删除的主键集合
     * @param isValid 是否进行有效性校验
     * @return 是否删除成功
     */
    @Override
    public Boolean deleteWithValidByIds(Collection<Long> ids, Boolean isValid) {
        if(isValid){
            //TODO 做一些业务上的校验,判断是否需要校验
        }
        return baseMapper.deleteByIds(ids) > 0;
    }

    @Override
    public void handleExpertPapers(List<ZjkExpertPaper> zjkExpertPaperList, Long expertId) {
        baseMapper.delete(new LambdaQueryWrapper<ZjkExpertPaper>()
            .eq(ZjkExpertPaper::getExpertId, expertId)
        );
        if (zjkExpertPaperList != null && !zjkExpertPaperList.isEmpty()){
            zjkExpertPaperList.forEach(zjkExpertAwards -> {
                zjkExpertAwards.setExpertId(expertId);
            });
            baseMapper.insertOrUpdateBatch(zjkExpertPaperList);
        }

    }

    @Override
    public List<ZjkExpertPaper> populateExpertPapersList(Long expertId) {
        List<ZjkExpertPaper> zjkExpertPaperList = baseMapper.selectList(
            new LambdaQueryWrapper<ZjkExpertPaper>()
                .eq(ZjkExpertPaper::getExpertId, expertId)
        );
        return zjkExpertPaperList.isEmpty() ? new ArrayList<>() : zjkExpertPaperList;
    }

    @Override
    public void deleteByProductId(Long productReviewId) {
        LambdaQueryWrapper<ZjkExpertPaper> wrapper = Wrappers.lambdaQuery();
        wrapper.eq(ZjkExpertPaper::getProductId, productReviewId);
        baseMapper.delete(wrapper);
    }

    @Override
    public boolean saveBatch(List<ZjkExpertPaper> zjkExpertPaperList) {
       return baseMapper.insertBatch(zjkExpertPaperList);
    }

    @Override
    public List<ZjkExpertPaper> selectListByProductIds(Long zjkProductReviewId) {
        LambdaQueryWrapper<ZjkExpertPaper> wrapper = Wrappers.lambdaQuery();
        wrapper.in(ZjkExpertPaper::getProductId, zjkProductReviewId);
       return baseMapper.selectList(wrapper);
    }

    @Override
    public List<ZjkExpertPaperVo> paperList(Long productId, String reviewSn) {
        return baseMapper.paperList(productId,reviewSn);
    }

    @Override
    public Long getExpertPaperCount() {
        return baseMapper.getExpertPaperCount();
    }


    @Override
    public List<ZjkExpertPaper> selectByExpertId(Long expertId) {
        return baseMapper.selectByExpertId(expertId);
    }
}
