package org.dromara.zjk.service.impl;

import cn.hutool.core.date.DateUtil;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import lombok.RequiredArgsConstructor;
import org.dromara.common.core.domain.R;
import org.dromara.common.core.exception.ServiceException;
import org.dromara.common.core.utils.MapstructUtils;
import org.dromara.common.core.utils.StringUtils;
import org.dromara.common.mybatis.core.page.PageQuery;
import org.dromara.common.mybatis.core.page.TableDataInfo;
import org.dromara.common.satoken.utils.LoginHelper;
import org.dromara.zjk.domain.*;
import org.dromara.zjk.domain.bo.ZjkCreditBo;
import org.dromara.zjk.domain.bo.ZjkCreditUserBo;
import org.dromara.zjk.domain.dto.CreditInfoDto;
import org.dromara.zjk.domain.vo.*;
import org.dromara.zjk.mapper.*;
import org.dromara.zjk.service.IZjkCreditUserService;
import org.springframework.stereotype.Service;

import java.time.LocalDate;
import java.time.temporal.ChronoUnit;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 信用用户管理Service业务层处理
 *
 * @author ygh
 * @date 2024-11-11
 */
@RequiredArgsConstructor
@Service
public class ZjkCreditUserServiceImpl implements IZjkCreditUserService {

    private final ZjkCreditUserMapper baseMapper;

    private final ZjkCreditRuleMapper zjkCreditRuleMapper;

    private final ZjkCreditMapper zjkCreditMapper;

    private final ZjkExpertMapper zjkExpertMapper;

    private final ZjkCreditLvMapper zjkCreditLvMapper;

    private final ZjkCreditRewardMapper zjkCreditRewardMapper;

    private final ZjkExpertItemMapper zjkExpertItemMapper;

    private final ZjkEvaluateMapper zjkEvaluateMapper;

    /**
     * 查询信用用户管理
     *
     * @param ID 主键
     * @return 信用用户管理
     */
    @Override
    public ZjkCreditUserVo queryById(String ID){
        return baseMapper.selectVoById(ID);
    }

    /**
     * 分页查询信用用户管理列表
     *
     * @param bo        查询条件
     * @param pageQuery 分页参数
     * @return 信用用户管理分页列表
     */
    @Override
    public TableDataInfo<ZjkCreditUserVo> queryPageList(ZjkCreditUserBo bo, PageQuery pageQuery) {
        LambdaQueryWrapper<ZjkCreditUser> lqw = buildQueryWrapper(bo);
        lqw.orderBy(true,false,ZjkCreditUser::getCreditCount);
        Page<ZjkCreditUserVo> result = baseMapper.selectVoPage(pageQuery.build(), lqw);
        List<ZjkCreditUserVo> records = result.getRecords();
        for (ZjkCreditUserVo record : records) {
            ZjkExpert zjkExpert = zjkExpertMapper.selectById(record.getExpertId());
            record.setZjkExpert(zjkExpert);
            ZjkCreditLvVo zjkCreditLvVo = zjkCreditLvMapper.selectVoById(record.getLvId());
            record.setZjkCreditLvVo(zjkCreditLvVo);
        }
        return TableDataInfo.build(result);
    }

    /**
     * 查询符合条件的信用用户管理列表
     *
     * @param bo 查询条件
     * @return 信用用户管理列表
     */
    @Override
    public List<ZjkCreditUserVo> queryList(ZjkCreditUserBo bo) {
        LambdaQueryWrapper<ZjkCreditUser> lqw = buildQueryWrapper(bo);
        return baseMapper.selectVoList(lqw);
    }

    private LambdaQueryWrapper<ZjkCreditUser> buildQueryWrapper(ZjkCreditUserBo bo) {
        Map<String, Object> params = bo.getParams();
        LambdaQueryWrapper<ZjkCreditUser> lqw = Wrappers.lambdaQuery();
        lqw.eq(StringUtils.isNotBlank(bo.getID()), ZjkCreditUser::getID, bo.getID());
        lqw.eq(bo.getUserId() != null, ZjkCreditUser::getUserId, bo.getUserId());
        lqw.eq(bo.getExpertId() != null, ZjkCreditUser::getExpertId, bo.getExpertId());
        lqw.eq(StringUtils.isNotBlank(bo.getRewardId()), ZjkCreditUser::getRewardId, bo.getRewardId());
        lqw.eq(StringUtils.isNotBlank(bo.getRewardTitle()), ZjkCreditUser::getRewardTitle, bo.getRewardTitle());
        lqw.eq(bo.getCreateBy() != null, ZjkCreditUser::getCreateBy, bo.getCreateBy());
        lqw.eq(bo.getCreateTime() != null, ZjkCreditUser::getCreateTime, bo.getCreateTime());
        lqw.eq(bo.getUpdateBy() != null, ZjkCreditUser::getUpdateBy, bo.getUpdateBy());
        lqw.eq(bo.getUpdateTime() != null, ZjkCreditUser::getUpdateTime, bo.getUpdateTime());
        lqw.eq(bo.getChangeTime() != null, ZjkCreditUser::getChangeTime, bo.getChangeTime());
        return lqw;
    }

    /**
     * 新增信用用户管理
     *
     * @param bo 信用用户管理
     * @return 是否新增成功
     */
    @Override
    public Boolean insertByBo(ZjkCreditUserBo bo) {
        ZjkCreditUser add = MapstructUtils.convert(bo, ZjkCreditUser.class);
        validEntityBeforeSave(add);
        boolean flag = baseMapper.insert(add) > 0;
        if (flag) {
            bo.setID(add.getID());
        }
        return flag;
    }

    /**
     * 修改信用用户管理
     *
     * @param bo 信用用户管理
     * @return 是否修改成功
     */
    @Override
    public Boolean updateByBo(ZjkCreditUserBo bo) {
        ZjkCreditUser update = MapstructUtils.convert(bo, ZjkCreditUser.class);
        validEntityBeforeSave(update);
        return baseMapper.updateById(update) > 0;
    }

    /**
     * 保存前的数据校验
     */
    private void validEntityBeforeSave(ZjkCreditUser entity){
        //TODO 做一些数据校验,如唯一约束
    }

    /**
     * 校验并批量删除信用用户管理信息
     *
     * @param ids     待删除的主键集合
     * @param isValid 是否进行有效性校验
     * @return 是否删除成功
     */
    @Override
    public Boolean deleteWithValidByIds(Collection<String> ids, Boolean isValid) {
        if(isValid){
            //TODO 做一些业务上的校验,判断是否需要校验
        }
        return baseMapper.deleteByIds(ids) > 0;
    }

    @Override
    public Boolean saveCreditUser(CreditInfoDto creditInfoDto) {
        //根据专家id获取信用信息
        LambdaQueryWrapper<ZjkCreditUser> lambdaQueryWrapper = new LambdaQueryWrapper<>();
        lambdaQueryWrapper.eq(ZjkCreditUser::getExpertId,creditInfoDto.getExpertId());
        lambdaQueryWrapper.orderBy(true,true,ZjkCreditUser::getUpdateTime).last("Limit 1");
        ZjkCreditUser zjkCreditUser = baseMapper.selectOne(lambdaQueryWrapper);
        if(zjkCreditUser!=null){

            //修改用户总信用
            zjkCreditUser.setExpertId(creditInfoDto.getExpertId());
            zjkCreditUser.setUserId(creditInfoDto.getUserId());
            zjkCreditUser.setChangeTime(new Date());
            //根据规则获取所得的信用分
            LambdaQueryWrapper<ZjkCreditRule> lambdaQueryWrapper1 = new LambdaQueryWrapper<>();
            lambdaQueryWrapper1.eq(ZjkCreditRule::getRuleKey,creditInfoDto.getRuleKey());
            ZjkCreditRuleVo zjkCreditRuleVo = zjkCreditRuleMapper.selectVoOne(lambdaQueryWrapper1);
            zjkCreditUser.setCreditCount(zjkCreditUser.getCreditCount()+zjkCreditRuleVo.getEachScore());
            //所有规则
            List<ZjkCreditReward> zjkCreditRewards = zjkCreditRewardMapper.selectList();
            Map<String, ZjkCreditReward> map = zjkCreditRewards.stream()
                .collect(Collectors.toMap(ZjkCreditReward::getID, Function.identity()));
//        专家信用惩罚03：信用分低于60；
            if(zjkCreditUser.getCreditCount()<60){
                ZjkCreditReward zjkCreditReward = map.get("3");
                if(zjkCreditUser.getRewardId()!=null){
                    String rewardId = zjkCreditUser.getRewardId();
                    String rewardTitle = zjkCreditUser.getRewardTitle();
                    if(!rewardId.contains("3")){
                        StringBuilder stringBuilder = new StringBuilder();
                        stringBuilder.append(rewardId).append(",").append("3");
                        stringBuilder = new StringBuilder();
                        stringBuilder.append(rewardTitle).append(zjkCreditReward.getRewardName());
                    }
                }else {
                    zjkCreditUser.setRewardId("3");
                    zjkCreditUser.setRewardTitle(zjkCreditReward.getRewardName());
                }
            }
            //获取专家的信用记录
            LambdaQueryWrapper<ZjkCredit> lambdaQueryWrapper2 = new LambdaQueryWrapper();
            lambdaQueryWrapper2.eq(ZjkCredit::getCreditId,zjkCreditUser.getExpertId());
//        一诺千金：连续5次接受过项目评审后，从未请假；
            //获取专家最近五次的评审状态，有没有五次，有没有请假，请了几次假，
            LambdaQueryWrapper<ZjkExpertItem> queryWrapper = new LambdaQueryWrapper<>();
            // 获取当前日期
            LocalDate now = LocalDate.now();
            // 减去一年
            LocalDate oneYearAgo = now.minus(1, ChronoUnit.YEARS);
            queryWrapper.ge(ZjkExpertItem::getCreateTime,oneYearAgo);
            queryWrapper.eq(ZjkExpertItem::getExpertId,zjkCreditUser.getExpertId());
            List<ZjkExpertItem> zjkExpertItems = zjkExpertItemMapper.selectList(queryWrapper);
            if(zjkExpertItems.size()>=5){
                int count = 0;
                //是否请假
                for (ZjkExpertItem zjkExpertItem : zjkExpertItems) {
                    String leaveStatus = zjkExpertItem.getLeaveStatus();
                    if(leaveStatus.equals(1)){
                        count++;
                    }
                }
                //        专家信用惩罚04：一年内5次项目请假；
                if(count>=5){
                    ZjkCreditReward zjkCreditReward = map.get("4");
                    zjkCreditUser.setRewardId("4");
                    zjkCreditUser.setRewardTitle(zjkCreditReward.getRewardName());
                }
                if(count==0){
                    ZjkCreditReward zjkCreditReward = map.get("1");
                    if(zjkCreditUser.getRewardId()!=null){
                        String rewardId = zjkCreditUser.getRewardId();
                        String rewardTitle = zjkCreditUser.getRewardTitle();
                        if(!rewardId.contains("1")){
                            StringBuilder stringBuilder = new StringBuilder();
                            stringBuilder.append(rewardId).append(",").append("1");
                            zjkCreditUser.setRewardId(stringBuilder.toString());
                            stringBuilder = new StringBuilder();
                            stringBuilder.append(rewardTitle).append(zjkCreditReward.getRewardName());
                            zjkCreditUser.setRewardTitle(stringBuilder.toString());
                        }
                    }else {
                        zjkCreditUser.setRewardId("1");
                        zjkCreditUser.setRewardTitle(zjkCreditReward.getRewardName());
                    }
                }
            }
//        信用大师：信用分在95以上持续6个月；
            Date sixMonthsAgo = new Date(System.currentTimeMillis() - 6 * 30 * 24 * 60 * 60 * 1000);
            lambdaQueryWrapper2.ge(ZjkCredit::getUpdateTime,sixMonthsAgo);
            lambdaQueryWrapper2.le(ZjkCredit::getCreditCount,94);
            List<ZjkCredit> zjkCredits = zjkCreditMapper.selectList(lambdaQueryWrapper2);
            if(zjkCredits!=null&&zjkCredits.size()>0){
                ZjkCreditReward zjkCreditReward = map.get("2");
                if(zjkCreditUser.getRewardId()!=null){
                    String rewardId = zjkCreditUser.getRewardId();
                    String rewardTitle = zjkCreditUser.getRewardTitle();
                    if(!rewardId.contains("2")){
                        StringBuilder stringBuilder = new StringBuilder();
                        stringBuilder.append(rewardId).append(",").append("2");
                        zjkCreditUser.setRewardId(stringBuilder.toString());
                        stringBuilder = new StringBuilder();
                        stringBuilder.append(rewardTitle).append(zjkCreditReward.getRewardName());
                        zjkCreditUser.setRewardTitle(stringBuilder.toString());
                    }
                }else {
                    zjkCreditUser.setRewardId("2");
                    zjkCreditUser.setRewardTitle(zjkCreditReward.getRewardName());
                }
            }
//        专家信用惩罚05：一年内，两次项目评审无故缺席（评价为未出席）；
            LambdaQueryWrapper<ZjkEvaluate> lambdaQueryWrapper3 = new LambdaQueryWrapper();
            lambdaQueryWrapper3.ge(ZjkEvaluate::getCreateTime,oneYearAgo);
            lambdaQueryWrapper3.eq(ZjkEvaluate::getIsPresent,0);
            lambdaQueryWrapper3.eq(ZjkEvaluate::getSourceRole,0);
            lambdaQueryWrapper3.eq(ZjkEvaluate::getStatus,1);
            List<ZjkEvaluate> zjkEvaluates = zjkEvaluateMapper.selectList(lambdaQueryWrapper3);
            if(zjkEvaluates.size()>=2){
                ZjkCreditReward zjkCreditReward = map.get("5");
                zjkCreditUser.setRewardId("5");
                zjkCreditUser.setRewardTitle(zjkCreditReward.getRewardName());
            }
            long creditCount = zjkCreditUser.getCreditCount();
            creditCount = Math.min(Math.max(creditCount, 0), 100);
            zjkCreditUser.setCreditCount(creditCount);
            if (creditCount > 80) {
                zjkCreditUser.setLvId(1L);
            } else if (creditCount > 60) {
                zjkCreditUser.setLvId(2L);
            } else if (creditCount > 40) {
                zjkCreditUser.setLvId(3L);
            } else if (creditCount > 20) {
                zjkCreditUser.setLvId(4L);
            } else {
                zjkCreditUser.setLvId(5L);
            }
            baseMapper.updateById(zjkCreditUser);
            //添加信用变更记录
            ZjkCredit zjkCredit = new ZjkCredit();
            zjkCredit.setCredit(creditInfoDto.getCreditSum());
            zjkCredit.setUserId(creditInfoDto.getUserId());
            zjkCredit.setCreditCount(zjkCreditUser.getCreditCount());
            zjkCredit.setExpertId(creditInfoDto.getExpertId());
            zjkCredit.setRuleId(zjkCreditRuleVo.getID());
            return zjkCreditMapper.insert(zjkCredit)>0;
        }else {
            return false;
        }
    }

    @Override
    public R<CreditExpertRankVO> getExpertPointRank(Integer rangeDay) {
        Date daysAgo = DateUtil.offsetDay(new Date(), -rangeDay+1);
        List<CreditExpertGetVO> res = baseMapper.getExpertPointRank(daysAgo);
        CreditExpertRankVO pointExpertRankVO = new CreditExpertRankVO();
        List<String> pointY = res.stream().map(CreditExpertGetVO::getExpertName).collect(Collectors.toList());
        List<Long> pointX = res.stream()
            .map(CreditExpertGetVO::getCreditCount)
            .map(count -> count > 100? 100 : count)
            .collect(Collectors.toList());
        pointExpertRankVO.setNameY(pointY);
        pointExpertRankVO.setPointX(pointX);
        return R.ok(pointExpertRankVO);
    }

    @Override
    public R<List<ZjkCreditRuleVo>> selectUserById(ZjkCreditBo zjkCreditBo, PageQuery pageQuery) {
        try {
            int count = 0;
            LambdaQueryWrapper<ZjkCreditRule> queryWrapper = new LambdaQueryWrapper<>();
            if(zjkCreditBo.getIntegralReason()!=null){
                queryWrapper.like(ZjkCreditRule::getRuleName,zjkCreditBo.getIntegralReason());
                count++;
            }
            List<Long> vos = new ArrayList<>();
            if(count!=0){
                List<ZjkCreditRuleVo> zjkCreditRuleVos = zjkCreditRuleMapper.selectVoList(queryWrapper);
                if(zjkCreditRuleVos==null||zjkCreditRuleVos.size()<=0){
                    return R.ok();
                }
                vos = zjkCreditRuleVos.stream().map(ZjkCreditRuleVo::getRuleKey).collect(Collectors.toList());
            }
            LambdaQueryWrapper<ZjkCredit> lambdaQueryWrapper = new LambdaQueryWrapper<>();
            Long expertId = zjkCreditBo.getExpertId();

            lambdaQueryWrapper.eq(ZjkCredit::getExpertId,expertId==null?LoginHelper.getUserId():expertId);
             if(zjkCreditBo.getStartTime()!=null&&zjkCreditBo.getEndTime()!=null){
                lambdaQueryWrapper.ge(ZjkCredit::getUpdateTime,zjkCreditBo.getStartTime());
                lambdaQueryWrapper.le(ZjkCredit::getUpdateTime,zjkCreditBo.getEndTime());
            }
             if(vos!=null&&vos.size()>0){
                 lambdaQueryWrapper.in(ZjkCredit::getRuleId,vos);
             }
            Page<ZjkCreditVo> zjkCreditVos = zjkCreditMapper.selectVoPage(pageQuery.build(), lambdaQueryWrapper);
//            List<ZjkCreditVo> zjkCreditVos = zjkCreditMapper.selectVoList(lambdaQueryWrapper);
             //所偶有的信用分规则
            List<ZjkCreditRuleVo> zjkCreditRuleVos = zjkCreditRuleMapper.selectVoList();
            Map<Long, ZjkCreditRuleVo> userMap = zjkCreditRuleVos.stream()
                .collect(Collectors.toMap(ZjkCreditRuleVo::getID, user -> user));
            List<ZjkCreditRuleVo> zjkCreditRuleVos1 = new ArrayList<>();
            for (ZjkCreditVo zjkCreditVo : zjkCreditVos.getRecords()) {
                if(zjkCreditVo.getRuleId()!=null){
                    ZjkCreditRuleVo zjkCreditRuleVo = userMap.get(zjkCreditVo.getRuleId());
                    zjkCreditRuleVo.setEachScore(zjkCreditVo.getCredit());
                    zjkCreditRuleVo.setUpdateTime(zjkCreditVo.getUpdateTime());
                    zjkCreditRuleVos1.add(zjkCreditRuleVo);
                }
            }
            return R.ok(zjkCreditRuleVos1);
        }catch (Exception e){
            return R.fail();
        }
    }

    @Override
    public R<List<ChartBaseCreditVO>> blacklist(ZjkCreditBo creditBo) {
        //根据名称获取专家id
        LambdaQueryWrapper<ZjkExpert> wrapper = new LambdaQueryWrapper<>();
        List<Long> collect1 = new ArrayList<>();
        int count = 0;
        if(creditBo.getExpertName()!=null){
            wrapper.like(ZjkExpert::getExpertName,creditBo.getExpertName());
            count++;
        }
        if(creditBo.getWorkUnit()!=null){
            wrapper.like(ZjkExpert::getWorkUnit,creditBo.getWorkUnit());
            count++;
        }
        if(creditBo.getStartTime()!=null&&creditBo.getEndTime()!=null){
            wrapper.ge(ZjkExpert::getApprovalTime, creditBo.getStartTime());
            wrapper.le(ZjkExpert::getApprovalTime, creditBo.getEndTime());
            count++;
        }
        if(count!=0){
            List<ZjkExpertVo> vos = zjkExpertMapper.selectVoList(wrapper);
            if(vos==null||vos.size()<=0){
                return R.ok();
            }
            collect1 = vos.stream().map(ZjkExpertVo::getExpertId).collect(Collectors.toList());
        }
        //获取所有行用分低于
        LambdaQueryWrapper<ZjkCreditUser> lambdaQueryWrapper = new LambdaQueryWrapper<>();
        lambdaQueryWrapper.ge(ZjkCreditUser::getRewardId,4);
        if(collect1!=null&&collect1.size()>0){
            lambdaQueryWrapper.in(ZjkCreditUser::getExpertId,collect1);
        }
        List<ZjkCreditUser> list = baseMapper.selectList(lambdaQueryWrapper);
        Map<Long, ZjkCreditUser> userMap = list.stream()
            .collect(Collectors.toMap(ZjkCreditUser::getExpertId, user -> user));
        List<Long> idList = list.stream()
            .map(ZjkCreditUser::getExpertId)
            .collect(Collectors.toList());
        List<ChartBaseCreditVO> list1 = new ArrayList<>();
        //所有黑名单专家
        if(idList!=null&&idList.size()>0) {
            List<ZjkExpert> zjkExperts = zjkExpertMapper.selectByIds(idList);
            Map<Long, ZjkExpert> collect = zjkExperts.stream()
                .collect(Collectors.toMap(ZjkExpert::getExpertId, zjkExpert -> zjkExpert));
            List<ZjkCreditLv> zjkCreditLvs = zjkCreditLvMapper.selectList();
            Map<Long, ZjkCreditLv> ZzkCreditLvMap = zjkCreditLvs.stream()
                .collect(Collectors.toMap(ZjkCreditLv::getId, user -> user));
            for (ZjkCreditUser zjkCreditUser : list) {
                ChartBaseCreditVO chartBaseCreditVO = new ChartBaseCreditVO();
                //专家信息
                ZjkExpert zjkExpert = collect.get(zjkCreditUser.getExpertId());
                if (zjkExpert != null) {
                    chartBaseCreditVO.setWorkUnit(zjkExpert.getWorkUnit());
                    chartBaseCreditVO.setDuty(zjkExpert.getDuty());
                    chartBaseCreditVO.setApprovalTime(zjkExpert.getApprovalTime());
                    chartBaseCreditVO.setBackParkName("信用分不满足要求");
                    chartBaseCreditVO.setExpertId(zjkCreditUser.getExpertId());
                    chartBaseCreditVO.setCreditCount(zjkCreditUser.getCreditCount());
                    chartBaseCreditVO.setLvName(ZzkCreditLvMap.get(zjkCreditUser.getLvId()).getCreditName());
                    chartBaseCreditVO.setExpertName(collect.get(zjkCreditUser.getExpertId()).getExpertName());
                    list1.add(chartBaseCreditVO);
                }

            }
        }
        return R.ok(list1);
    }

    @Override
    public Boolean creditId(ZjkCreditUserBo bo) {
        ZjkCreditUser zjkCreditUser = baseMapper.selectOne(new LambdaQueryWrapper<ZjkCreditUser>()
            .eq(ZjkCreditUser::getExpertId,bo.getExpertId()));
        zjkCreditUser.setRewardId("");
        zjkCreditUser.setRewardTitle("");
        return baseMapper.updateById(zjkCreditUser)>0;
    }

    @Override
    public Boolean moveToBlacklist(ZjkCreditUserBo bo) {
        ZjkCreditUser zjkCreditUser = baseMapper.selectOne(new LambdaQueryWrapper<ZjkCreditUser>()
            .eq(ZjkCreditUser::getUserId,bo.getUserId())
            .eq(ZjkCreditUser::getExpertId,bo.getExpertId()));
        if(zjkCreditUser==null){
            throw new ServiceException("该专家信用数据异常");
        }
        zjkCreditUser.setRewardId("6");
        zjkCreditUser.setRewardTitle(LoginHelper.getUsername()+"手动移入黑名单");
        return baseMapper.updateById(zjkCreditUser)>0;
    }

    @Override
    public Boolean removeTheBlacklist(ZjkCreditUserBo bo) {
        ZjkCreditUser zjkCreditUser = baseMapper.selectOne(new LambdaQueryWrapper<ZjkCreditUser>()
            .eq(ZjkCreditUser::getUserId,bo.getUserId())
            .eq(ZjkCreditUser::getExpertId,bo.getExpertId()));
        if(zjkCreditUser==null){
            throw new ServiceException("该专家信用数据异常");
        }
        zjkCreditUser.setRewardId(null);
        zjkCreditUser.setRewardTitle(null);

        return baseMapper.removeTheBlacklist(zjkCreditUser)>0;
    }

    @Override
    public Long getBlacklistCount() {
        LambdaQueryWrapper<ZjkCreditUser> wrapper = Wrappers.lambdaQuery();
        wrapper.ge(ZjkCreditUser::getRewardId, 4); // 获取 id 大于等于 4 的数据
        return baseMapper.selectCount(wrapper);
    }

    @Override
    public Long getCount() {
        LambdaQueryWrapper<ZjkCreditUser> wrapper = Wrappers.lambdaQuery();
        wrapper.eq(ZjkCreditUser::getLvId, 1);
        return baseMapper.selectCount(wrapper);
    }
}
