package org.dromara.common.core.utils;

import lombok.AccessLevel;
import lombok.NoArgsConstructor;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.dromara.common.core.domain.DateRange;

import java.lang.management.ManagementFactory;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.*;
import java.time.format.DateTimeFormatter;
import java.util.Date;

/**
 * 时间工具类
 *
 * @author ruoyi
 */
@NoArgsConstructor(access = AccessLevel.PRIVATE)
public class DateUtils extends org.apache.commons.lang3.time.DateUtils {

    public static final String YYYY = "yyyy";

    public static final String YYYY_MM = "yyyy-MM";

    public static final String YYYY_MM_DD = "yyyy-MM-dd";

    public static final String YYYYMMDDHHMMSS = "yyyyMMddHHmmss";

    public static final String YYYY_MM_DD_HH_MM_SS = "yyyy-MM-dd HH:mm:ss";

    private static final String[] PARSE_PATTERNS = {
        "yyyy-MM-dd", "yyyy-MM-dd HH:mm:ss", "yyyy-MM-dd HH:mm", "yyyy-MM",
        "yyyy/MM/dd", "yyyy/MM/dd HH:mm:ss", "yyyy/MM/dd HH:mm", "yyyy/MM",
        "yyyy.MM.dd", "yyyy.MM.dd HH:mm:ss", "yyyy.MM.dd HH:mm", "yyyy.MM"};

    /**
     * 获取当前Date型日期
     *
     * @return Date() 当前日期
     */
    public static Date getNowDate() {
        return new Date();
    }

    /**
     * 获取当前日期, 默认格式为yyyy-MM-dd
     *
     * @return String
     */
    public static String getDate() {
        return dateTimeNow(YYYY_MM_DD);
    }

    public static String getTime() {
        return dateTimeNow(YYYY_MM_DD_HH_MM_SS);
    }

    public static String dateTimeNow() {
        return dateTimeNow(YYYYMMDDHHMMSS);
    }

    public static String dateTimeNow(final String format) {
        return parseDateToStr(format, new Date());
    }

    public static String dateTime(final Date date) {
        return parseDateToStr(YYYY_MM_DD, date);
    }

    public static String parseDateToStr(final String format, final Date date) {
        return new SimpleDateFormat(format).format(date);
    }

    public static Date dateTime(final String format, final String ts) {
        try {
            return new SimpleDateFormat(format).parse(ts);
        } catch (ParseException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * 日期路径 即年/月/日 如2018/08/08
     */
    public static String datePath() {
        Date now = new Date();
        return DateFormatUtils.format(now, "yyyy/MM/dd");
    }

    /**
     * 日期路径 即年/月/日 如20180808
     */
    public static String dateTime() {
        Date now = new Date();
        return DateFormatUtils.format(now, "yyyyMMdd");
    }

    /**
     * 日期型字符串转化为日期 格式
     */
    public static Date parseDate(Object str) {
        if (str == null) {
            return null;
        }
        try {
            return parseDate(str.toString(), PARSE_PATTERNS);
        } catch (ParseException e) {
            return null;
        }
    }

    /**
     * 获取服务器启动时间
     */
    public static Date getServerStartDate() {
        long time = ManagementFactory.getRuntimeMXBean().getStartTime();
        return new Date(time);
    }

    /**
     * 计算相差天数
     */
    public static int differentDaysByMillisecond(Date date1, Date date2) {
        return Math.abs((int) ((date2.getTime() - date1.getTime()) / (1000 * 3600 * 24)));
    }

    /**
     * 计算两个时间差
     */
    public static String getDatePoor(Date endDate, Date nowDate) {
        long nd = 1000 * 24 * 60 * 60;
        long nh = 1000 * 60 * 60;
        long nm = 1000 * 60;
        // long ns = 1000;
        // 获得两个时间的毫秒时间差异
        long diff = endDate.getTime() - nowDate.getTime();
        // 计算差多少天
        long day = diff / nd;
        // 计算差多少小时
        long hour = diff % nd / nh;
        // 计算差多少分钟
        long min = diff % nd % nh / nm;
        // 计算差多少秒//输出结果
        // long sec = diff % nd % nh % nm / ns;
        return day + "天" + hour + "小时" + min + "分钟";
    }

    /**
     * 增加 LocalDateTime ==> Date
     */
    public static Date toDate(LocalDateTime temporalAccessor) {
        ZonedDateTime zdt = temporalAccessor.atZone(ZoneId.systemDefault());
        return Date.from(zdt.toInstant());
    }

    /**
     * 增加 LocalDate ==> Date
     */
    public static Date toDate(LocalDate temporalAccessor) {
        LocalDateTime localDateTime = LocalDateTime.of(temporalAccessor, LocalTime.of(0, 0, 0));
        ZonedDateTime zdt = localDateTime.atZone(ZoneId.systemDefault());
        return Date.from(zdt.toInstant());
    }

    public static String getFirstAndLastDayOfMonth(int month) {
        // 获取当前年份
        int currentYear = LocalDate.now().getYear();

        // 创建日期格式化器
        DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy.MM.dd");

        // 获取该月的第一天
        LocalDate firstDay = LocalDate.of(currentYear, month, 1);
        // 获取该月的最后一天
        LocalDate lastDay = firstDay.withDayOfMonth(firstDay.lengthOfMonth());

        // 格式化日期并拼接
        String formattedFirstDay = firstDay.format(formatter);
        String formattedLastDay = lastDay.format(formatter);

        return formattedFirstDay + "~" + formattedLastDay;
    }
    /**
     * 判断日期是否在给定区间内
     *
     * @param date      要检查的日期
     * @param startDate 区间起始日期（包含）
     * @param endDate   区间结束日期（包含）
     * @return 如果日期在区间内，返回 true；否则返回 false
     */
    public static boolean isWithinLeavePeriod(Date date, Date startDate, Date endDate) {
        return (startDate == null || !date.before(startDate)) && // 起始日期为空或 date >= startDate
            (endDate == null || !date.after(endDate));       // 结束日期为空或 date <= endDate
    }

    private static Date localDateToDate(LocalDateTime localDateTime) {
        return Date.from(localDateTime.atZone(ZoneId.systemDefault()).toInstant());
    }



    /**
     * 返回指定天数的日期范围
     *
     * @param days 天数（如30、60、90）
     * @return 包含开始日期（指定天数前）和结束日期（当前日期）的 DateRange
     */
    public static DateRange getDateRange(int days) {
        // 当前日期
        LocalDate now = LocalDate.now();
        // 指定天数前的日期
        LocalDate pastDate = now.minusDays(days);

        // 转换为 java.util.Date 类型
        Date startDate = localDateToDate(pastDate.atStartOfDay());
        Date endDate = localDateToDate(LocalDateTime.of(now, LocalTime.MAX)); // 设置为当天的 23:59:59


        return new DateRange(startDate,endDate);
    }

    /**
     * 获取当前月份的开始时间和结束时间，返回格式化后的字符串数组
     * @return 当前月份的开始时间和结束时间 [startDate, endDate]
     */
    public static String[] getCurrentMonthDateRange() {
        // 定义日期格式化器
        DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy-MM-dd");

        // 获取当前月第一天的开始时间
        LocalDate startOfMonth = LocalDate.now().withDayOfMonth(1);
        // 获取当前月最后一天的结束时间
        LocalDate endOfMonth = LocalDate.now().withDayOfMonth(LocalDate.now().lengthOfMonth());

        // 格式化为字符串
        String startDate = startOfMonth.format(formatter);
        String endDate = endOfMonth.format(formatter);

        return new String[]{startDate, endDate};
    }
    /**
     * 获取上个月的开始时间和结束时间
     *
     * @return 上个月的开始时间和结束时间 ["yyyy-MM-dd", "yyyy-MM-dd"]
     */
    public static Date[] getLastMonthDateRange(LocalDate localDate) {
        // 获取当前日期，并计算上个月
        LocalDate now = localDate;
        LocalDate firstDayOfLastMonth = now.minusMonths(1).withDayOfMonth(1); // 上个月的第一天
        LocalDate lastDayOfLastMonth = firstDayOfLastMonth.withDayOfMonth(firstDayOfLastMonth.lengthOfMonth()); // 上个月的最后一天

        // 将 LocalDate 转换为 LocalDateTime，设置为上个月的第一天的 00:00:00 和最后一天的 23:59:59
        LocalDateTime firstDayDateTime = firstDayOfLastMonth.atStartOfDay(); // 第一天下午 00:00:00
        LocalDateTime lastDayDateTime = lastDayOfLastMonth.atTime(23, 59, 59); // 最后一天的 23:59:59

        // 指定时区
        ZoneId zone = ZoneId.systemDefault();

        // 将 LocalDateTime 转换为 ZonedDateTime
        ZonedDateTime firstDayZonedDateTime = firstDayDateTime.atZone(zone);
        ZonedDateTime lastDayZonedDateTime = lastDayDateTime.atZone(zone);

        // 将 ZonedDateTime 转换为 Instant
        Instant firstDayInstant = firstDayZonedDateTime.toInstant();
        Instant lastDayInstant = lastDayZonedDateTime.toInstant();

        // 将 Instant 转换为 Date
        Date firstDayDate = Date.from(firstDayInstant);
        Date lastDayDate = Date.from(lastDayInstant);

        return new Date[]{firstDayDate, lastDayDate};
    }

    /**
     * 获取上一年第一天和最后一天的时间范围
     *
     * @param localDate 当前日期
     * @return 上一年的第一天和最后一天的时间范围
     */
    public static Date[] getLastYearDateRange(LocalDate localDate) {
        // 获取当前日期的年份
        int year = localDate.getYear();

        // 获取上一年的第一天和最后一天
        LocalDate firstDayOfLastYear = LocalDate.of(year - 1, 1, 1); // 上一年第一天
        LocalDate lastDayOfLastYear = LocalDate.of(year - 1, 12, 31); // 上一年最后一天

        // 将 LocalDate 转换为 LocalDateTime，设置为上一年第一天的 00:00:00 和最后一天的 23:59:59
        LocalDateTime firstDayDateTime = firstDayOfLastYear.atStartOfDay(); // 第一天下午 00:00:00
        LocalDateTime lastDayDateTime = lastDayOfLastYear.atTime(23, 59, 59); // 最后一天的 23:59:59

        // 指定时区
        ZoneId zone = ZoneId.systemDefault();

        // 将 LocalDateTime 转换为 ZonedDateTime
        ZonedDateTime firstDayZonedDateTime = firstDayDateTime.atZone(zone);
        ZonedDateTime lastDayZonedDateTime = lastDayDateTime.atZone(zone);

        // 将 ZonedDateTime 转换为 Instant
        Instant firstDayInstant = firstDayZonedDateTime.toInstant();
        Instant lastDayInstant = lastDayZonedDateTime.toInstant();

        // 将 Instant 转换为 Date
        Date firstDayDate = Date.from(firstDayInstant);
        Date lastDayDate = Date.from(lastDayInstant);

        return new Date[]{firstDayDate, lastDayDate};
    }
}
