package org.dromara.zjk.annualInspection.annualInspectionAudit.controller;

import cn.dev33.satoken.annotation.SaCheckPermission;
import cn.dev33.satoken.annotation.SaCheckRole;
import cn.dev33.satoken.annotation.SaMode;
import jakarta.servlet.http.HttpServletResponse;
import jakarta.validation.constraints.NotEmpty;
import jakarta.validation.constraints.NotNull;
import lombok.RequiredArgsConstructor;
import org.dromara.common.core.constant.TenantConstants;
import org.dromara.common.core.domain.R;
import org.dromara.common.core.utils.StringUtils;
import org.dromara.common.core.validate.AddGroup;
import org.dromara.common.core.validate.EditGroup;
import org.dromara.common.excel.utils.ExcelUtil;
import org.dromara.common.idempotent.annotation.RepeatSubmit;
import org.dromara.common.log.annotation.Log;
import org.dromara.common.log.enums.BusinessType;
import org.dromara.common.mybatis.core.page.PageQuery;
import org.dromara.common.mybatis.core.page.TableDataInfo;
import org.dromara.common.web.core.BaseController;
import org.dromara.zjk.annualInspection.annualInspectionAudit.domain.bo.ZjkAnnualInspectionBo;
import org.dromara.zjk.annualInspection.annualInspectionAudit.domain.vo.ZjkAnnualInspectionVo;
import org.dromara.zjk.annualInspection.annualInspectionAudit.service.IZjkAnnualInspectionService;
import org.dromara.zjk.enums.AnnualInspectionStatus;
import org.dromara.zjk.zwy.annotation.DecryptOperation;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * 年检管理
 * 前端访问路由地址为:/expert/annualInspection
 *
 * @author zhangzhou
 * @date 2024-11-13
 */
@Validated
@RequiredArgsConstructor
@RestController
@RequestMapping("/annualInspectionAudit")
public class ZjkAnnualInspectionController extends BaseController {

    private final IZjkAnnualInspectionService zjkAnnualInspectionService;

    /**
     * 查询年检管理列表
     */
    @DecryptOperation
    @SaCheckRole(value = {
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY
    }, mode = SaMode.OR)
    @GetMapping("/list")
    public TableDataInfo<ZjkAnnualInspectionVo> list(ZjkAnnualInspectionBo bo, PageQuery pageQuery) {
        return zjkAnnualInspectionService.queryPageList(bo, pageQuery);
    }

    /**
     * 导出年检管理列表
     */
    @SaCheckRole(value = {
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY
    }, mode = SaMode.OR)
    @Log(title = "年检管理", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public void export(ZjkAnnualInspectionBo bo, HttpServletResponse response) {
        List<ZjkAnnualInspectionVo> list = zjkAnnualInspectionService.queryList(bo);
        ExcelUtil.exportExcel(list, "年检管理", ZjkAnnualInspectionVo.class, response);
    }

    /**
     * 获取年检管理详细信息
     *
     * @param id 主键
     */
    @SaCheckRole(value = {
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY
    }, mode = SaMode.OR)
    @GetMapping("/{id}")
    public R<ZjkAnnualInspectionVo> getInfo(@NotNull(message = "主键不能为空")
                                     @PathVariable Long id) {
        return R.ok(zjkAnnualInspectionService.queryById(id));
    }

    /**
     * 新增年检管理
     */
    @SaCheckPermission("annualInspection:annualInspectionAudit:add")
    @Log(title = "年检管理", businessType = BusinessType.INSERT)
    @RepeatSubmit()
    @PostMapping("/add")
    public R<Void> add(@Validated(AddGroup.class) @RequestBody ZjkAnnualInspectionBo bo) {
        return toAjax(zjkAnnualInspectionService.insertByBo(bo));
    }

    /**
     * 修改年检管理
     */
    @SaCheckPermission("annualInspection:annualInspectionAudit:edit")
    @Log(title = "年检管理", businessType = BusinessType.UPDATE)
    @RepeatSubmit()
    @PostMapping("/edit")
    public R<Void> edit(@Validated(EditGroup.class) @RequestBody ZjkAnnualInspectionBo bo) {
        return toAjax(zjkAnnualInspectionService.updateByBo(bo));
    }

    /**
     * 删除年检管理
     *
     * @param ids 主键串
     */
    @SaCheckPermission("annualInspection:annualInspectionAudit:remove")
    @Log(title = "年检管理", businessType = BusinessType.DELETE)
    @GetMapping("/delete/{ids}")
    public R<Void> remove(@NotEmpty(message = "主键不能为空")
                          @PathVariable Long[] ids) {
        return toAjax(zjkAnnualInspectionService.deleteWithValidByIds(List.of(ids), true));
    }

    /**
     * 批量年检
     */
    @SaCheckRole(value = {
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY
    }, mode = SaMode.OR)
    @Log(title = "年检管理", businessType = BusinessType.UPDATE)
    @PostMapping("/batchAnnualInspection")
    public R<Void> batchAnnualInspection(@RequestBody ZjkAnnualInspectionBo zjkAnnualInspectionBo) {
        // 检查参数有效性
        if (zjkAnnualInspectionBo.getIds() == null || zjkAnnualInspectionBo.getIds().isEmpty()) {
            return R.fail("请选择需要年检的数据");
        }
        if (!AnnualInspectionStatus.PASSED.getCode().equals(zjkAnnualInspectionBo.getAnnualInspectionStatus()) &&
            !AnnualInspectionStatus.FAILED.getCode().equals(zjkAnnualInspectionBo.getAnnualInspectionStatus())) {
            return R.fail("请选择有效审核结果");
        }
        if (AnnualInspectionStatus.FAILED.getCode().equals(zjkAnnualInspectionBo.getAnnualInspectionStatus())&& StringUtils.isBlank(zjkAnnualInspectionBo.getReason())) {
            return R.fail("请输入驳回理由");
        }
        // 执行批量年检操作
        try {
            zjkAnnualInspectionService.batchAnnualInspection(zjkAnnualInspectionBo);
        } catch (Exception e) {
            e.printStackTrace();
            return R.fail("操作失败");
        }
        return R.ok();
    }


}
