package org.dromara.zjk.annualInspection.annualInspectionRecords.controller;

import cn.dev33.satoken.annotation.SaCheckPermission;
import cn.dev33.satoken.annotation.SaCheckRole;
import cn.dev33.satoken.annotation.SaMode;
import jakarta.servlet.http.HttpServletResponse;
import jakarta.validation.constraints.NotEmpty;
import jakarta.validation.constraints.NotNull;
import lombok.RequiredArgsConstructor;
import org.dromara.common.core.constant.TenantConstants;
import org.dromara.common.core.domain.R;
import org.dromara.common.core.validate.AddGroup;
import org.dromara.common.core.validate.EditGroup;
import org.dromara.common.excel.utils.ExcelUtil;
import org.dromara.common.idempotent.annotation.RepeatSubmit;
import org.dromara.common.log.annotation.Log;
import org.dromara.common.log.enums.BusinessType;
import org.dromara.common.mybatis.core.page.PageQuery;
import org.dromara.common.mybatis.core.page.TableDataInfo;
import org.dromara.common.web.core.BaseController;
import org.dromara.zjk.annualInspection.annualInspectionRecords.domain.bo.ZjkAnnualInspectionRecordsBo;
import org.dromara.zjk.annualInspection.annualInspectionRecords.domain.vo.ZjkAnnualInspectionRecordsVo;
import org.dromara.zjk.annualInspection.annualInspectionRecords.service.IZjkAnnualInspectionRecordsService;
import org.dromara.zjk.zwy.annotation.DecryptOperation;
import org.dromara.zjk.zwy.annotation.EncryptOperation;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * 年检记录
 * 前端访问路由地址为:/annualInspection/annualInspectionRecords
 *
 * @author zhangzhou
 * @date 2024-11-13
 */
@Validated
@RequiredArgsConstructor
@RestController
@RequestMapping("/annualInspectionRecords")
public class ZjkAnnualInspectionRecordsController extends BaseController {

    private final IZjkAnnualInspectionRecordsService zjkAnnualInspectionRecordsService;

    /**
     * 查询年检记录列表
     */
    @SaCheckRole(value = {
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY
    }, mode = SaMode.OR)
    @DecryptOperation
    @GetMapping("/list")
    public TableDataInfo<ZjkAnnualInspectionRecordsVo> list(ZjkAnnualInspectionRecordsBo bo, PageQuery pageQuery) {
        return zjkAnnualInspectionRecordsService.queryPageList(bo, pageQuery);
    }

    /**
     * 导出年检记录列表
     */
    @SaCheckPermission("annualInspection:annualInspectionRecords:export")
    @Log(title = "年检记录", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public void export(ZjkAnnualInspectionRecordsBo bo, HttpServletResponse response) {
        List<ZjkAnnualInspectionRecordsVo> list = zjkAnnualInspectionRecordsService.queryList(bo);
        ExcelUtil.exportExcel(list, "年检记录", ZjkAnnualInspectionRecordsVo.class, response);
    }

    /**
     * 获取年检记录详细信息
     *
     * @param id 主键
     */
    @DecryptOperation
    @SaCheckPermission("annualInspection:annualInspectionRecords:query")
    @GetMapping("/{id}")
    public R<ZjkAnnualInspectionRecordsVo> getInfo(@NotNull(message = "主键不能为空")
                                     @PathVariable Long id) {
        return R.ok(zjkAnnualInspectionRecordsService.queryById(id));
    }

    /**
     * 新增年检记录
     */
    @EncryptOperation
    @SaCheckPermission("annualInspection:annualInspectionRecords:add")
    @Log(title = "年检记录", businessType = BusinessType.INSERT)
    @RepeatSubmit()
    @PostMapping("/add")
    public R<Void> add(@Validated(AddGroup.class) @RequestBody ZjkAnnualInspectionRecordsBo bo) {
        return toAjax(zjkAnnualInspectionRecordsService.insertByBo(bo));
    }

    /**
     * 修改年检记录
     */
    @EncryptOperation
    @SaCheckPermission("annualInspection:annualInspectionRecords:edit")
    @Log(title = "年检记录", businessType = BusinessType.UPDATE)
    @RepeatSubmit()
    @PostMapping("/edit")
    public R<Void> edit(@Validated(EditGroup.class) @RequestBody ZjkAnnualInspectionRecordsBo bo) {
        return toAjax(zjkAnnualInspectionRecordsService.updateByBo(bo));
    }

    /**
     * 删除年检记录
     *
     * @param ids 主键串
     */
    @SaCheckPermission("annualInspection:annualInspectionRecords:remove")
    @Log(title = "年检记录", businessType = BusinessType.DELETE)
    @GetMapping("/delete/{ids}")
    public R<Void> remove(@NotEmpty(message = "主键不能为空")
                          @PathVariable Long[] ids) {
        return toAjax(zjkAnnualInspectionRecordsService.deleteWithValidByIds(List.of(ids), true));
    }
}
