package org.dromara.zjk.controller;

import java.util.List;

import cn.dev33.satoken.annotation.SaCheckRole;
import cn.dev33.satoken.annotation.SaMode;
import lombok.RequiredArgsConstructor;
import jakarta.servlet.http.HttpServletResponse;
import jakarta.validation.constraints.*;
import cn.dev33.satoken.annotation.SaCheckPermission;
import org.dromara.common.core.constant.TenantConstants;
import org.springframework.web.bind.annotation.*;
import org.springframework.validation.annotation.Validated;
import org.dromara.common.idempotent.annotation.RepeatSubmit;
import org.dromara.common.log.annotation.Log;
import org.dromara.common.web.core.BaseController;
import org.dromara.common.mybatis.core.page.PageQuery;
import org.dromara.common.core.domain.R;
import org.dromara.common.core.validate.AddGroup;
import org.dromara.common.core.validate.EditGroup;
import org.dromara.common.log.enums.BusinessType;
import org.dromara.common.excel.utils.ExcelUtil;
import org.dromara.zjk.domain.vo.LearningEvaluateVo;
import org.dromara.zjk.domain.bo.LearningEvaluateBo;
import org.dromara.zjk.service.ILearningEvaluateService;
import org.dromara.common.mybatis.core.page.TableDataInfo;

/**
 * 学习园地-学习评估
 * 前端访问路由地址为:/zjk/learningEvaluate
 *
 * @author ygh
 * @date 2025-07-28
 */
@Validated
@RequiredArgsConstructor
@RestController
@RequestMapping("/learningEvaluate")
public class LearningEvaluateController extends BaseController {

    private final ILearningEvaluateService learningEvaluateService;

    /**
     * 查询学习园地-学习评估列表
     */
    @SaCheckRole(value = {
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY
    },mode = SaMode.OR)
    @GetMapping("/list")
    public TableDataInfo<LearningEvaluateVo> list(LearningEvaluateBo bo, PageQuery pageQuery) {
        return learningEvaluateService.queryPageList(bo, pageQuery);
    }


    /**
     * 获取学习园地-学习评估详细信息
     *
     * @param ID 主键
     */
    @SaCheckRole(value = {
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY
    },mode = SaMode.OR)
    @GetMapping("/{ID}")
    public R<LearningEvaluateVo> getInfo(@NotNull(message = "主键不能为空")
                                     @PathVariable String ID) {
        return R.ok(learningEvaluateService.queryById(ID));
    }

    /**
     * 新增学习园地-学习评估
     */
    @SaCheckRole(value = {
        TenantConstants.EXPERT_ROLE_KEY
    },mode = SaMode.OR)
    @Log(title = "学习园地-学习评估", businessType = BusinessType.INSERT)
    @RepeatSubmit()
    @PostMapping("/add")
    public R<Void> add(@Validated(AddGroup.class) @RequestBody LearningEvaluateBo bo) {
        return toAjax(learningEvaluateService.insertByBo(bo));
    }

    /**
     * 修改学习园地-学习评估
     */
    @SaCheckRole(value = {
        TenantConstants.EXPERT_ROLE_KEY
    },mode = SaMode.OR)
    @Log(title = "学习园地-学习评估", businessType = BusinessType.UPDATE)
    @RepeatSubmit()
    @PostMapping("/edit")
    public R<Void> edit(@Validated(EditGroup.class) @RequestBody LearningEvaluateBo bo) {
        return toAjax(learningEvaluateService.updateByBo(bo));
    }

    /**
     * 删除学习园地-学习评估
     *
     * @param IDs 主键串
     */
    @SaCheckRole(value = {
        TenantConstants.EXPERT_ROLE_KEY
    },mode = SaMode.OR)
    @Log(title = "学习园地-学习评估", businessType = BusinessType.DELETE)
    @GetMapping("/delete/{IDs}")
    public R<Void> remove(@NotEmpty(message = "主键不能为空")
                          @PathVariable String[] IDs) {
        return toAjax(learningEvaluateService.deleteWithValidByIds(List.of(IDs), true));
    }
}
