package org.dromara.zjk.controller;

import java.util.List;

import cn.dev33.satoken.annotation.SaCheckRole;
import cn.dev33.satoken.annotation.SaMode;
import lombok.RequiredArgsConstructor;
import jakarta.servlet.http.HttpServletResponse;
import jakarta.validation.constraints.*;
import cn.dev33.satoken.annotation.SaCheckPermission;
import org.dromara.common.core.constant.TenantConstants;
import org.dromara.zjk.service.ILearningPlanService;
import org.springframework.web.bind.annotation.*;
import org.springframework.validation.annotation.Validated;
import org.dromara.common.idempotent.annotation.RepeatSubmit;
import org.dromara.common.log.annotation.Log;
import org.dromara.common.web.core.BaseController;
import org.dromara.common.mybatis.core.page.PageQuery;
import org.dromara.common.core.domain.R;
import org.dromara.common.core.validate.AddGroup;
import org.dromara.common.core.validate.EditGroup;
import org.dromara.common.log.enums.BusinessType;
import org.dromara.common.excel.utils.ExcelUtil;
import org.dromara.zjk.domain.vo.LearningPlanVo;
import org.dromara.zjk.domain.bo.LearningPlanBo;
import org.dromara.common.mybatis.core.page.TableDataInfo;

/**
 * 学习计划
 * 前端访问路由地址为:/zjk/learningPlan
 *
 * @author ygh
 * @date 2025-07-28
 */
@Validated
@RequiredArgsConstructor
@RestController
@RequestMapping("/learningPlan")
public class LearningPlanController extends BaseController {

    private final ILearningPlanService learningPlanService;

    /**
     * 查询专家学习计划列表
     */
    @SaCheckRole(value = {
        TenantConstants.EXPERT_ROLE_KEY
    },mode = SaMode.OR)
    @GetMapping("/list")
    public TableDataInfo<LearningPlanVo> list(LearningPlanBo bo, PageQuery pageQuery) {
        return learningPlanService.queryPageList(bo, pageQuery);
    }



    /**
     * 获取专家学习计划详细信息
     *
     * @param ID 主键
     */
    @SaCheckRole(value = {
        TenantConstants.EXPERT_ROLE_KEY
    },mode = SaMode.OR)
    @GetMapping("/{ID}")
    public R<LearningPlanVo> getInfo(@NotNull(message = "主键不能为空")
                                     @PathVariable String ID) {
        return R.ok(learningPlanService.queryById(ID));
    }

    /**
     * 新增专家学习计划
     */
    @SaCheckRole(value = {
        TenantConstants.EXPERT_ROLE_KEY
    },mode = SaMode.OR)
    @Log(title = "学习计划新增", businessType = BusinessType.INSERT)
    @RepeatSubmit()
    @PostMapping("/add")
    public R<Void> add(@Validated(AddGroup.class) @RequestBody LearningPlanBo bo) {
        return toAjax(learningPlanService.insertByBo(bo));
    }

    /**
     * 修改专家学习计划
     */
    @SaCheckRole(value = {
        TenantConstants.EXPERT_ROLE_KEY
    },mode = SaMode.OR)
    @Log(title = "学习计划修改", businessType = BusinessType.UPDATE)
    @RepeatSubmit()
    @PostMapping("/edit")
    public R<Void> edit(@Validated(EditGroup.class) @RequestBody LearningPlanBo bo) {
        return toAjax(learningPlanService.updateByBo(bo));
    }

    /**
     * 删除专家学习计划
     *
     * @param IDs 主键串
     */
    @SaCheckRole(value = {
        TenantConstants.EXPERT_ROLE_KEY
    },mode = SaMode.OR)
    @Log(title = "学习计划删除", businessType = BusinessType.DELETE)
    @GetMapping("/delete/{IDs}")
    public R<Void> remove(@NotEmpty(message = "主键不能为空")
                          @PathVariable String[] IDs) {
        return toAjax(learningPlanService.deleteWithValidByIds(List.of(IDs), true));
    }
}
