package org.dromara.zjk.controller;

import java.util.List;

import cn.dev33.satoken.annotation.SaCheckRole;
import cn.dev33.satoken.annotation.SaMode;
import lombok.RequiredArgsConstructor;
import jakarta.servlet.http.HttpServletResponse;
import jakarta.validation.constraints.*;
import cn.dev33.satoken.annotation.SaCheckPermission;
import org.dromara.common.core.constant.TenantConstants;
import org.springframework.web.bind.annotation.*;
import org.springframework.validation.annotation.Validated;
import org.dromara.common.idempotent.annotation.RepeatSubmit;
import org.dromara.common.log.annotation.Log;
import org.dromara.common.web.core.BaseController;
import org.dromara.common.mybatis.core.page.PageQuery;
import org.dromara.common.core.domain.R;
import org.dromara.common.core.validate.AddGroup;
import org.dromara.common.core.validate.EditGroup;
import org.dromara.common.log.enums.BusinessType;
import org.dromara.common.excel.utils.ExcelUtil;
import org.dromara.zjk.domain.vo.LearningSuggestVo;
import org.dromara.zjk.domain.bo.LearningSuggestBo;
import org.dromara.zjk.service.ILearningSuggestService;
import org.dromara.common.mybatis.core.page.TableDataInfo;

/**
 * 学习园地-学习推荐
 * 前端访问路由地址为:/zjk/learningSuggest
 *
 * @author ygh
 * @date 2025-07-28
 */
@Validated
@RequiredArgsConstructor
@RestController
@RequestMapping("/learningSuggest")
public class LearningSuggestController extends BaseController {

    private final ILearningSuggestService learningSuggestService;

    /**
     * 查询学习园地-学习推荐列表
     */
    @SaCheckRole(value = {

        TenantConstants.JG_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY
    },mode = SaMode.OR)
    @GetMapping("/list")
    public TableDataInfo<LearningSuggestVo> list(LearningSuggestBo bo, PageQuery pageQuery) {
        return learningSuggestService.queryPageList(bo, pageQuery);
    }

    /**
     * 学习园地-学习推荐历史记录列表
     */
    @SaCheckRole(value = {
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "学习园地-学习推荐", businessType = BusinessType.VIEW)
    @GetMapping("/myList")
    public R<List<LearningSuggestVo>> mySuggests(LearningSuggestBo bo) {
        List<LearningSuggestVo> list = learningSuggestService.queryList(bo);
       return  R.ok( list);
    }

    /**
     * 获取学习园地-学习推荐详细信息
     *
     * @param ID 主键
     */
    @SaCheckRole(value = {
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY
    },mode = SaMode.OR)
    @GetMapping("/{ID}")
    public R<LearningSuggestVo> getInfo(@NotNull(message = "主键不能为空")
                                     @PathVariable String ID) {
        return R.ok(learningSuggestService.queryById(ID));
    }

    /**
     * 新增学习园地-学习推荐
     */
    @SaCheckRole(value = {
        TenantConstants.EXPERT_ROLE_KEY
    },mode = SaMode.OR)
    @Log(title = "学习园地-学习推荐", businessType = BusinessType.INSERT)
    @RepeatSubmit()
    @PostMapping("/add")
    public R<Void> add(@Validated(AddGroup.class) @RequestBody LearningSuggestBo bo) {
        return toAjax(learningSuggestService.insertByBo(bo));
    }

    /**
     * 修改学习园地-学习推荐
     */
    @SaCheckRole(value = {
        TenantConstants.EXPERT_ROLE_KEY
    },mode = SaMode.OR)
    @Log(title = "学习园地-学习推荐", businessType = BusinessType.UPDATE)
    @RepeatSubmit()
    @PostMapping("/edit")
    public R<Void> edit(@Validated(EditGroup.class) @RequestBody LearningSuggestBo bo) {
        return toAjax(learningSuggestService.updateByBo(bo));
    }

    /**
     * 删除学习园地-学习推荐
     *
     * @param IDs 主键串
     */
    @SaCheckRole(value = {
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY
    },mode = SaMode.OR)
    @Log(title = "学习园地-学习推荐", businessType = BusinessType.DELETE)
    @GetMapping("/delete/{IDs}")
    public R<Void> remove(@NotEmpty(message = "主键不能为空")
                          @PathVariable String[] IDs) {
        return toAjax(learningSuggestService.deleteWithValidByIds(List.of(IDs), true));
    }
}
