package org.dromara.zjk.controller;

import cn.dev33.satoken.annotation.SaCheckRole;
import cn.dev33.satoken.annotation.SaMode;
import jakarta.servlet.http.HttpServletResponse;
import jakarta.validation.constraints.NotNull;
import org.dromara.common.core.constant.TenantConstants;
import org.dromara.common.core.domain.R;
import org.dromara.common.excel.utils.ExcelUtil;
import org.dromara.common.idempotent.annotation.RepeatSubmit;
import org.dromara.common.log.annotation.Log;
import org.dromara.common.log.enums.BusinessType;
import org.dromara.common.mybatis.core.page.PageQuery;
import org.dromara.common.mybatis.core.page.TableDataInfo;
import org.dromara.common.web.core.BaseController;
import org.dromara.zjk.domain.vo.StsEvaluateNumVO;
import org.dromara.zjk.domain.vo.StsEvaluateVO;
import org.dromara.zjk.domain.vo.StsListVO;
import org.dromara.zjk.domain.vo.ZjkEvaluateVO;
import org.dromara.zjk.service.ZjkEvaluateService;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import java.util.List;

/**
 * 专家库专家互评控制器
 *
 * @author : yanguohua
 * @version : 1.0
 * @date : 2024/10/30 16:57
 */
@RestController
@RequestMapping("/evaluate")
public class ZjkEvaluateController extends BaseController {
    @Resource
    private ZjkEvaluateService zjkEvaluateService;


    /**
     * 查询专家评价列表-专家
     */
    @GetMapping("/listToZj")
    @SaCheckRole(TenantConstants.EXPERT_ROLE_KEY)
    public TableDataInfo<ZjkEvaluateVO> listToZj(ZjkEvaluateVO bo, PageQuery pageQuery) {
        return zjkEvaluateService.getListToZj(bo, pageQuery);
    }

    /**
     * 查询专家评价列表-管理员
     */
    @GetMapping("/listToOrg")
    @SaCheckRole(value = {
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY
    }, mode = SaMode.OR)
    public TableDataInfo<ZjkEvaluateVO> listToOrg(ZjkEvaluateVO bo, PageQuery pageQuery) {
        return zjkEvaluateService.getListToOrg(bo, pageQuery);
    }

    /**
     * 查询专家评价列表-项目管理员
     */
    @GetMapping("/listToXm")
    @SaCheckRole(value = {
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
    }, mode = SaMode.OR
    )
    public TableDataInfo<ZjkEvaluateVO> listToXm(ZjkEvaluateVO bo, PageQuery pageQuery) {
        return zjkEvaluateService.getListToXm(bo, pageQuery);
    }

    /**
     * 专家评价新增
     */
    @Log(title = "专家评价", businessType = BusinessType.INSERT)
    @RepeatSubmit()
    @PostMapping()
    @SaCheckRole(value = {
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY
    }, mode = SaMode.OR)
    public R<Void> addComment(@RequestBody ZjkEvaluateVO bo) {
        return zjkEvaluateService.addComment(bo);
    }

    /**
     * 获取专家评价详细信息
     *
     * @param gid 主键
     */
    @GetMapping("/detail/{gid}")
    @SaCheckRole(value = {
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY
    }, mode = SaMode.OR)
    public R<ZjkEvaluateVO> getInfo(@NotNull(message = "主键不能为空")
                                    @PathVariable String gid) {
        return R.ok(zjkEvaluateService.getInfoById(gid));
    }


    /**
     * 评价反馈
     */
    @GetMapping("/getEvaluateListByExpert")
    @SaCheckRole(value = {
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY
    }, mode = SaMode.OR)
    public TableDataInfo<ZjkEvaluateVO> getEvaluateListByExpert(ZjkEvaluateVO bo, PageQuery pageQuery) {
        return zjkEvaluateService.getEvaluateListByExpert(bo, pageQuery);
    }

    /**
     * 获取评价统计次数
     */
    @GetMapping("/stsNum")
    @SaCheckRole(
        TenantConstants.JG_ROLE_KEY
    )
    public R<StsEvaluateNumVO> getStsNum() {
        return R.ok(zjkEvaluateService.getStsNum());
    }

    /**
     * 专家互评图表统计
     */
    @GetMapping("/stsChart")
    @SaCheckRole(
        TenantConstants.JG_ROLE_KEY
    )
    public R<StsEvaluateVO> stsChart(Integer rangeMonth) {
        return zjkEvaluateService.getStsChart(rangeMonth);
    }

    /**
     * 互评统计列表-管理员
     */
    @GetMapping("/stsList")
    @SaCheckRole(
        TenantConstants.JG_ROLE_KEY
    )
    public TableDataInfo<StsListVO> getStsList(ZjkEvaluateVO bo, PageQuery pageQuery) {
        return zjkEvaluateService.getStsList(bo, pageQuery);
    }

    /**
     * 互评统计导出
     */
    @Log(title = "评价统计导出", businessType = BusinessType.EXPORT)
    @SaCheckRole(
        TenantConstants.JG_ROLE_KEY
    )
    @PostMapping("/export")
    public void export(ZjkEvaluateVO bo, HttpServletResponse response) {
        List<StsListVO> list = zjkEvaluateService.selectStsEvaluateList(bo);
        ExcelUtil.exportExcel(list, "评价统计", StsListVO.class, response);
    }
}
