package org.dromara.zjk.controller;

import cn.dev33.satoken.annotation.SaCheckPermission;
import jakarta.servlet.http.HttpServletResponse;
import jakarta.validation.constraints.NotEmpty;
import jakarta.validation.constraints.NotNull;
import lombok.RequiredArgsConstructor;
import org.dromara.common.core.domain.R;
import org.dromara.common.core.validate.AddGroup;
import org.dromara.common.core.validate.EditGroup;
import org.dromara.common.excel.utils.ExcelUtil;
import org.dromara.common.idempotent.annotation.RepeatSubmit;
import org.dromara.common.log.annotation.Log;
import org.dromara.common.log.enums.BusinessType;
import org.dromara.common.mybatis.core.page.PageQuery;
import org.dromara.common.mybatis.core.page.TableDataInfo;
import org.dromara.common.web.core.BaseController;
import org.dromara.zjk.domain.bo.ZjkExpertHistoryBo;
import org.dromara.zjk.domain.vo.ZjkExpertHistoryVo;
import org.dromara.zjk.service.IZjkExpertHistoryService;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * 专家记录变更历史表
 * 前端访问路由地址为:/expert/expertHistory
 *
 * @author zhangzhou
 * @date 2024-11-02
 */
@Validated
@RequiredArgsConstructor
@RestController
@RequestMapping("/expertHistory")
public class ZjkExpertHistoryController extends BaseController {

    private final IZjkExpertHistoryService zjkExpertHistoryService;

    /**
     * 查询专家记录变更历史表列表
     */
    @SaCheckPermission("expert:expertHistory:list")
    @GetMapping("/list")
    public TableDataInfo<ZjkExpertHistoryVo> list(ZjkExpertHistoryBo bo, PageQuery pageQuery) {
        return zjkExpertHistoryService.queryPageList(bo, pageQuery);
    }

    /**
     * 导出专家记录变更历史表列表
     */
    @SaCheckPermission("expert:expertHistory:export")
    @Log(title = "专家记录变更历史表", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public void export(ZjkExpertHistoryBo bo, HttpServletResponse response) {
        List<ZjkExpertHistoryVo> list = zjkExpertHistoryService.queryList(bo);
        ExcelUtil.exportExcel(list, "专家记录变更历史表", ZjkExpertHistoryVo.class, response);
    }

    /**
     * 获取专家记录变更历史表详细信息
     *
     * @param id 主键
     */
    @SaCheckPermission("expert:expertHistory:query")
    @GetMapping("/{id}")
    public R<ZjkExpertHistoryVo> getInfo(@NotNull(message = "主键不能为空")
                                     @PathVariable Long id) {
        return R.ok(zjkExpertHistoryService.queryById(id));
    }

    /**
     * 新增专家记录变更历史表
     */
    @SaCheckPermission("expert:expertHistory:add")
    @Log(title = "专家记录变更历史表", businessType = BusinessType.INSERT)
    @RepeatSubmit()
    @PostMapping()
    public R<Void> add(@Validated(AddGroup.class) @RequestBody ZjkExpertHistoryBo bo) {
        return toAjax(zjkExpertHistoryService.insertByBo(bo));
    }

    /**
     * 修改专家记录变更历史表
     */
    @SaCheckPermission("expert:expertHistory:edit")
    @Log(title = "专家记录变更历史表", businessType = BusinessType.UPDATE)
    @RepeatSubmit()
    @PutMapping()
    public R<Void> edit(@Validated(EditGroup.class) @RequestBody ZjkExpertHistoryBo bo) {
        return toAjax(zjkExpertHistoryService.updateByBo(bo));
    }

    /**
     * 删除专家记录变更历史表
     *
     * @param ids 主键串
     */
    @SaCheckPermission("expert:expertHistory:remove")
    @Log(title = "专家记录变更历史表", businessType = BusinessType.DELETE)
    @DeleteMapping("/{ids}")
    public R<Void> remove(@NotEmpty(message = "主键不能为空")
                          @PathVariable Long[] ids) {
        return toAjax(zjkExpertHistoryService.deleteWithValidByIds(List.of(ids), true));
    }
}
