package org.dromara.zjk.controller;

import cn.dev33.satoken.annotation.SaCheckPermission;
import cn.dev33.satoken.annotation.SaCheckRole;
import cn.dev33.satoken.annotation.SaMode;
import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.collection.CollectionUtil;
import com.alibaba.fastjson.JSON;
import jakarta.servlet.http.HttpServletResponse;
import jakarta.validation.constraints.NotEmpty;
import jakarta.validation.constraints.NotNull;
import lombok.RequiredArgsConstructor;
import org.dromara.common.core.constant.TenantConstants;
import org.dromara.common.core.domain.R;
import org.dromara.common.core.validate.AddGroup;
import org.dromara.common.excel.utils.ExcelUtil;
import org.dromara.common.idempotent.annotation.RepeatSubmit;
import org.dromara.common.log.annotation.Log;
import org.dromara.common.log.enums.BusinessType;
import org.dromara.common.mybatis.core.page.PageQuery;
import org.dromara.common.mybatis.core.page.TableDataInfo;
import org.dromara.common.web.core.BaseController;
import org.dromara.zjk.domain.ZjkExpertItem;
import org.dromara.zjk.domain.ZjkItemExtreation;
import org.dromara.zjk.domain.bo.ZjkExpertItemBo;
import org.dromara.zjk.domain.bo.ZjkItemExtreationBo;
import org.dromara.zjk.domain.bo.ZjkProjectExpertItemBo;
import org.dromara.zjk.domain.dto.SmsUpEventDto;
import org.dromara.zjk.domain.vo.ZjkExpertItemVo;
import org.dromara.zjk.domain.vo.ZjkExpertVo;
import org.dromara.zjk.domain.vo.ZjkItemExtreationVo;
import org.dromara.zjk.service.IZjkExpertItemService;
import org.dromara.zjk.zwy.annotation.DecryptOperation;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 *专家抽取记录
 * 前端访问路由地址为:/jyk/expertItem
 *
 * @author ygh
 * @date 2024-10-29
 */
@Validated
@RequiredArgsConstructor
@RestController
@RequestMapping("/project/extract")
public class ZjkExpertItemController extends BaseController {

    private final IZjkExpertItemService zjkExpertItemService;

    /**
     * 查询专家列表
     */
    @SaCheckPermission("project:extract:list")
    @GetMapping("/list")
    public TableDataInfo<ZjkItemExtreationVo> list(ZjkItemExtreationBo bo, PageQuery pageQuery) {
        return zjkExpertItemService.queryPageList(bo, pageQuery);
    }

    /**
     * 导出专家抽取列表
     */
    @SaCheckPermission("project:extract:export")
    @Log(title = "导出", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public void export(ZjkExpertItemBo bo, HttpServletResponse response) {
        List<ZjkExpertItemVo> list = zjkExpertItemService.queryList(bo);
        ExcelUtil.exportExcel(list, "导出", ZjkExpertItemVo.class, response);
    }

    /**
     * 获取专家抽取详细信息
     *
     * @param id 主键
     */
//    @SaCheckPermission("jyk:expertItem:query")
    @GetMapping("/{id}")
    public R<ZjkExpertItemVo> getInfo(@NotNull(message = "主键不能为空")
                                     @PathVariable Long id) {
        return R.ok(zjkExpertItemService.queryById(id));
    }

    /**
     * 新增专家抽取
     */
    @SaCheckPermission("project:extract:add")
    @Log(title = "新增", businessType = BusinessType.INSERT)
    @RepeatSubmit()
    @PostMapping("/add")
    public R<Void> add(@Validated(AddGroup.class) @RequestBody ZjkExpertItemBo bo) {
        return toAjax(zjkExpertItemService.insertByBo(bo));
    }

    /**
     * 修改专家抽取
     */
    @Log(title = "修改", businessType = BusinessType.UPDATE)
    @RepeatSubmit()
        @PostMapping("/update")
    public R<Void> edit(@RequestBody ZjkExpertItemBo bo) {
        return toAjax(zjkExpertItemService.updateByBo(bo));
    }

    /**
     * 删除专家抽取
     *
     * @param ids 主键串
     */
    @SaCheckPermission("project:extract:remove")
    @Log(title = "删除", businessType = BusinessType.DELETE)
    @PostMapping("/{ids}")
    public R<Void> remove(@NotEmpty(message = "主键不能为空")
                          @PathVariable Long[] ids) {
        return toAjax(zjkExpertItemService.deleteWithValidByIds(List.of(ids), true));
    }

    /**
     * 专家抽取
     *
     * @param zjkProjectExpertItemBo 抽取设置
     */
    @SaCheckPermission("project:extract:extract")
    @Log(title = "抽取", businessType = BusinessType.OTHER)
    @PostMapping("/expertltem")
    public R<Long> expertltem(@Validated(AddGroup.class) @RequestBody String zjkProjectExpertItemBo) {
        ZjkProjectExpertItemBo itemBo = JSON.parseObject(zjkProjectExpertItemBo, ZjkProjectExpertItemBo.class);
        return R.ok(zjkExpertItemService.expertltem(itemBo));
    }

    //专家级联查询
    @DecryptOperation
    @GetMapping ("/concatenated")
    public R<List<ZjkExpertVo>> concatenated(String name) {
        return R.ok(zjkExpertItemService.concatenated(name));
    }

    @GetMapping ("/unitList")
    public R<Set<String>> unitList(String name) {
        return R.ok(zjkExpertItemService.unitList(name));
    }

    @PostMapping ("/cancel")
    public R<Void> cancel(@Validated(AddGroup.class) @RequestBody ZjkProjectExpertItemBo itemExpertId) {
        zjkExpertItemService.cancel(itemExpertId);
        return R.ok();
    }
    @DecryptOperation
    @GetMapping ("/expert/top5")
    public R<List<ZjkExpertItemVo>> expertTop5(String name) {
        return R.ok(zjkExpertItemService.expertTop5());
    }

    /**
     * 通过 记录id 查询专家
     * @param reviewId
     * @return
     */
    @SaCheckRole(value = {
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
    },mode = SaMode.OR)
    @DecryptOperation
    @GetMapping("/get-expert")
    public R<List<ZjkExpertItemVo>> getByReviewId(Long reviewId) {
        return R.ok("success",zjkExpertItemService.getByReviewId(reviewId));
    }

    /**
     * 专家短信息回复回调接口
     * @param smsUpEventDto
     * @return
     */
    @PostMapping("/sms/reply")
    public R<String> handleSmsReply(@RequestBody SmsUpEventDto smsUpEventDto) {
        zjkExpertItemService.handleSmsReply(smsUpEventDto);
        return R.ok();
    }
}
