package org.dromara.zjk.controller;

import cn.dev33.satoken.annotation.SaCheckPermission;
import cn.dev33.satoken.annotation.SaCheckRole;
import cn.dev33.satoken.annotation.SaMode;
import jakarta.servlet.http.HttpServletResponse;
import jakarta.validation.constraints.NotEmpty;
import jakarta.validation.constraints.NotNull;
import lombok.RequiredArgsConstructor;
import org.dromara.common.core.constant.TenantConstants;
import org.dromara.common.core.domain.R;
import org.dromara.common.core.validate.AddGroup;
import org.dromara.common.core.validate.EditGroup;
import org.dromara.common.excel.utils.ExcelUtil;
import org.dromara.common.idempotent.annotation.RepeatSubmit;
import org.dromara.common.log.annotation.Log;
import org.dromara.common.log.enums.BusinessType;
import org.dromara.common.mybatis.core.page.PageQuery;
import org.dromara.common.mybatis.core.page.TableDataInfo;
import org.dromara.common.web.core.BaseController;
import org.dromara.zjk.domain.bo.ZjkExpertLogBo;
import org.dromara.zjk.domain.vo.ZjkExpertLogVo;
import org.dromara.zjk.service.IZjkExpertLogService;
import org.dromara.zjk.zwy.annotation.DecryptOperation;
import org.dromara.zjk.zwy.annotation.EncryptOperation;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.util.List;
import java.util.Map;

/**
 * 入库记录
 * 前端访问路由地址为:/expert/expertLog
 *
 * @author zhangzhou
 * @date 2024-10-31
 */
@Validated
@RequiredArgsConstructor
@RestController
@RequestMapping("/expertLog")
public class ZjkExpertLogController extends BaseController {

    private final IZjkExpertLogService zjkExpertLogService;

    /**
     * 查询入库记录列表
     */
    @DecryptOperation
    @SaCheckPermission("expert:expertLog:list")
    @GetMapping("/list")
    @SaCheckRole(value = {
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY
    }, mode = SaMode.OR)
    public TableDataInfo<ZjkExpertLogVo> list(ZjkExpertLogBo bo, PageQuery pageQuery) {
        return zjkExpertLogService.queryPageList(bo, pageQuery);
    }

    /**
     * 导出入库记录列表
     */
    @DecryptOperation
    @Log(title = "入库记录", businessType = BusinessType.EXPORT)
    @SaCheckRole(value = {
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY
    }, mode = SaMode.OR)
    @PostMapping("/export")
    public void export(ZjkExpertLogBo bo, HttpServletResponse response) {
        List<ZjkExpertLogVo> list = zjkExpertLogService.queryList(bo);
        ExcelUtil.exportExcel(list, "入库记录", ZjkExpertLogVo.class, response);
    }

    /**
     * 获取入库记录详细信息
     *
     * @param id 主键
     */
    @DecryptOperation
    @SaCheckPermission("expert:expertLog:query")
    @GetMapping("/{id}")
    @SaCheckRole(value = {
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY
    }, mode = SaMode.OR)
    public R<ZjkExpertLogVo> getInfo(@NotNull(message = "主键不能为空")
                                     @PathVariable Long id) {
        return R.ok(zjkExpertLogService.queryById(id));
    }

    /**
     * 新增入库记录
     */
    @EncryptOperation
    @SaCheckPermission("expert:expertLog:add")
    @Log(title = "入库记录", businessType = BusinessType.INSERT)
    @RepeatSubmit()
    @PostMapping()
    public R<Void> add(@Validated(AddGroup.class) @RequestBody ZjkExpertLogBo bo) {
        return toAjax(zjkExpertLogService.insertByBo(bo));
    }

    /**
     * 修改入库记录
     */
    @EncryptOperation
    @SaCheckPermission("expert:expertLog:edit")
    @Log(title = "入库记录", businessType = BusinessType.UPDATE)
    @RepeatSubmit()
    @PutMapping()
    public R<Void> edit(@Validated(EditGroup.class) @RequestBody ZjkExpertLogBo bo) {
        return toAjax(zjkExpertLogService.updateByBo(bo));
    }

    /**
     * 删除入库记录
     *
     * @param ids 主键串
     */
    @SaCheckPermission("expert:expertLog:remove")
    @Log(title = "入库记录", businessType = BusinessType.DELETE)
    @DeleteMapping("/{ids}")
    public R<Void> remove(@NotEmpty(message = "主键不能为空")
                          @PathVariable Long[] ids) {
        return toAjax(zjkExpertLogService.deleteWithValidByIds(List.of(ids), true));
    }


    /**
     * 趋势分析
     * @return
     */
    @SaCheckRole(value={
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY
    },mode = SaMode.OR)
    @GetMapping("getApplicationResults")
    public R<List<Map<String, Object>>> getApplicationResults(@RequestParam(name = "startTime",required = false)String startTime, @RequestParam(name = "endTime",required = false)String endTime) {
        return R.ok(zjkExpertLogService.getApplicationResults(startTime,endTime));
    }





}
