package org.dromara.zjk.controller;

import cn.dev33.satoken.annotation.SaCheckPermission;
import cn.dev33.satoken.annotation.SaCheckRole;
import cn.dev33.satoken.annotation.SaMode;
import jakarta.servlet.http.HttpServletResponse;
import jakarta.validation.constraints.NotEmpty;
import jakarta.validation.constraints.NotNull;
import lombok.RequiredArgsConstructor;
import org.dromara.common.core.constant.TenantConstants;
import org.dromara.common.core.domain.R;
import org.dromara.common.core.validate.AddGroup;
import org.dromara.common.core.validate.EditGroup;
import org.dromara.common.excel.utils.ExcelUtil;
import org.dromara.common.idempotent.annotation.RepeatSubmit;
import org.dromara.common.log.annotation.Log;
import org.dromara.common.log.enums.BusinessType;
import org.dromara.common.mybatis.core.page.PageQuery;
import org.dromara.common.mybatis.core.page.TableDataInfo;
import org.dromara.common.web.core.BaseController;
import org.dromara.zjk.domain.bo.ZjkExpertMajorBo;
import org.dromara.zjk.domain.vo.ZjkExpertMajorVo;
import org.dromara.zjk.service.IZjkExpertMajorService;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.util.List;
import java.util.Map;

/**
 * 专家库 专业关系表
 * 前端访问路由地址为:/expert/expertMajor
 *
 * @author ygh
 * @date 2024-11-01
 */
@Validated
@RequiredArgsConstructor
@RestController
@RequestMapping("/expertMajor")
public class ZjkExpertMajorController extends BaseController {

    private final IZjkExpertMajorService zjkExpertMajorService;

    /**
     * 查询专家库 专业关系表列表
     */
    @SaCheckPermission("expert:expertMajor:list")
    @GetMapping("/list")
    public TableDataInfo<ZjkExpertMajorVo> list(ZjkExpertMajorBo bo, PageQuery pageQuery) {
        return zjkExpertMajorService.queryPageList(bo, pageQuery);
    }

    /**
     * 导出专家库 专业关系表列表
     */
    @SaCheckPermission("expert:expertMajor:export")
    @Log(title = "专家库 专业关系表", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public void export(ZjkExpertMajorBo bo, HttpServletResponse response) {
        List<ZjkExpertMajorVo> list = zjkExpertMajorService.queryList(bo);
        ExcelUtil.exportExcel(list, "专家库 专业关系表", ZjkExpertMajorVo.class, response);
    }

    /**
     * 获取专家库 专业关系表详细信息
     *
     * @param id 主键
     */
    @SaCheckPermission("expert:expertMajor:query")
    @GetMapping("/{id}")
    public R<ZjkExpertMajorVo> getInfo(@NotNull(message = "主键不能为空")
                                     @PathVariable Long id) {
        return R.ok(zjkExpertMajorService.queryById(id));
    }

    /**
     * 新增专家库 专业关系表
     */
    @SaCheckPermission("expert:expertMajor:add")
    @Log(title = "专家库 专业关系表", businessType = BusinessType.INSERT)
    @RepeatSubmit()
    @PostMapping()
    public R<Void> add(@Validated(AddGroup.class) @RequestBody ZjkExpertMajorBo bo) {
        return toAjax(zjkExpertMajorService.insertByBo(bo));
    }

    /**
     * 修改专家库 专业关系表
     */
    @SaCheckPermission("expert:expertMajor:edit")
    @Log(title = "专家库 专业关系表", businessType = BusinessType.UPDATE)
    @RepeatSubmit()
    @PutMapping()
    public R<Void> edit(@Validated(EditGroup.class) @RequestBody ZjkExpertMajorBo bo) {
        return toAjax(zjkExpertMajorService.updateByBo(bo));
    }

    /**
     * 删除专家库 专业关系表
     *
     * @param ids 主键串
     */
    @SaCheckPermission("expert:expertMajor:remove")
    @Log(title = "专家库 专业关系表", businessType = BusinessType.DELETE)
    @DeleteMapping("/{ids}")
    public R<Void> remove(@NotEmpty(message = "主键不能为空")
                          @PathVariable Long[] ids) {
        return toAjax(zjkExpertMajorService.deleteWithValidByIds(List.of(ids), true));
    }

    /**
     * 获取专家各个学历的人数
     * @return
     */
    @SaCheckRole(value={
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY
    },mode = SaMode.OR)
    @GetMapping("getExpertMajorCount")
    public R<List<Map<Integer,Integer>> > getExpertMajorCount() {
        return R.ok(zjkExpertMajorService.getExpertMajorCount());
    }

}
