package org.dromara.zjk.controller;

import cn.dev33.satoken.annotation.SaCheckPermission;
import cn.dev33.satoken.annotation.SaCheckRole;
import cn.dev33.satoken.annotation.SaMode;
import cn.hutool.core.collection.CollUtil;
import jakarta.servlet.http.HttpServletResponse;
import jakarta.validation.constraints.NotEmpty;
import jakarta.validation.constraints.NotNull;
import lombok.RequiredArgsConstructor;
import org.dromara.common.core.constant.TenantConstants;
import org.dromara.common.core.domain.R;
import org.dromara.common.core.validate.AddGroup;
import org.dromara.common.core.validate.EditGroup;
import org.dromara.common.excel.utils.ExcelUtil;
import org.dromara.common.idempotent.annotation.RepeatSubmit;
import org.dromara.common.log.annotation.Log;
import org.dromara.common.log.enums.BusinessType;
import org.dromara.common.mybatis.core.page.PageQuery;
import org.dromara.common.mybatis.core.page.TableDataInfo;
import org.dromara.common.web.core.BaseController;
import org.dromara.zjk.domain.bo.ZjkExpertMoneyConfigBo;
import org.dromara.zjk.domain.excels.ZjkProductSummarizeExcelVO;
import org.dromara.zjk.domain.vo.SelectOptionVo;
import org.dromara.zjk.domain.vo.ZjkExpertMoneyConfigVo;
import org.dromara.zjk.service.IZjkExpertMoneyConfigService;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;

/**
 * 专家费用管理
 * 前端访问路由地址为:/expert/expertMoneyConfig
 *
 * @author by
 * @date 2024-11-25
 */
@Validated
@RequiredArgsConstructor
@RestController
@RequestMapping("/expertMoneyConfig")
public class ZjkExpertMoneyConfigController extends BaseController {

    private final IZjkExpertMoneyConfigService zjkExpertMoneyConfigService;

    /**
     * 查询专家费用管理列表
     */
    @SaCheckRole(value = {
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
    },mode = SaMode.OR)
    @GetMapping("/list")
    public TableDataInfo<ZjkExpertMoneyConfigVo> list(ZjkExpertMoneyConfigBo bo, PageQuery pageQuery) {
        return zjkExpertMoneyConfigService.queryPageList(bo, pageQuery);
    }

    /**
     * 导出专家费用管理列表
     */
    @SaCheckRole(value = {
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "专家费用管理", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public void export(ZjkExpertMoneyConfigBo bo, HttpServletResponse response) {
        List<ZjkExpertMoneyConfigVo> list = zjkExpertMoneyConfigService.queryList(bo);
        ExcelUtil.exportExcel(list, "专家费用管理", ZjkExpertMoneyConfigVo.class, response);
    }

    /**
     * 模板导出
     * @param response
     */
    @SaCheckRole(value = {
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "专家费用管理模板", businessType = BusinessType.EXPORT)
    @PostMapping("/export-template")
    public void exportTemplate(HttpServletResponse response) {
        ExcelUtil.exportExcel(new ArrayList<>(), "专家费用模板", ZjkExpertMoneyConfigVo.class, response);
    }

    /**
     * 导入
     * @param file
     * @return
     * @throws Exception
     */
    @SaCheckPermission("expert:expert:import")
    @Log(title = "专家库-专家信息导入", businessType = BusinessType.IMPORT)
    @PostMapping("/imports")
    @SaCheckRole(value = {
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY
    }, mode = SaMode.OR)
    public R imports(@RequestPart("file") MultipartFile file) throws Exception {
        InputStream inputStream = file.getInputStream();
        // 文件转化
        List<ZjkExpertMoneyConfigVo> vos = ExcelUtil.importExcel(inputStream, ZjkExpertMoneyConfigVo.class);

        Boolean imports = zjkExpertMoneyConfigService.imports(vos);
        return toAjax(imports);
    }

    /**
     * 获取专家费用管理详细信息
     *
     * @param moneyId 主键
     */
    @SaCheckRole(value = {
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
    },mode = SaMode.OR)
    @GetMapping("/{moneyId}")
    public R<ZjkExpertMoneyConfigVo> getInfo(@NotNull(message = "主键不能为空")
                                     @PathVariable Long moneyId) {
        return R.ok(zjkExpertMoneyConfigService.queryById(moneyId));
    }

    /**
     * 新增专家费用管理
     */
    @SaCheckRole(value = {
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "专家费用管理", businessType = BusinessType.INSERT)
    @RepeatSubmit()
    @PostMapping("/add")
    public R<Void> add(@Validated(AddGroup.class) @RequestBody ZjkExpertMoneyConfigBo bo) {
        return toAjax(zjkExpertMoneyConfigService.insertByBo(bo));
    }

    /**
     * 修改专家费用管理
     */
    @SaCheckRole(value = {
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "专家费用管理", businessType = BusinessType.UPDATE)
    @RepeatSubmit()
    @PostMapping("/edit")
    public R<Void> edit(@Validated(EditGroup.class) @RequestBody ZjkExpertMoneyConfigBo bo) {
        return toAjax(zjkExpertMoneyConfigService.updateByBo(bo));
    }

    /**
     * 删除专家费用管理
     *
     * @param moneyIds 主键串
     */
    @SaCheckRole(value = {
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "专家费用管理", businessType = BusinessType.DELETE)
    @GetMapping("/remove/{moneyIds}")
    public R<Void> remove(@NotEmpty(message = "主键不能为空")
                          @PathVariable Long[] moneyIds) {
        return toAjax(zjkExpertMoneyConfigService.deleteWithValidByIds(List.of(moneyIds), true));
    }


    /**
     * 获取职称等级下拉选项
     */
    @SaCheckRole(value = {
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
    },mode = SaMode.OR)
    @GetMapping("/getExpertMoneyConfig")
    public R<List<SelectOptionVo>> getExpertMoneyConfig() {
        return R.ok(zjkExpertMoneyConfigService.getExpertMoneyConfig());
    }

}
