package org.dromara.zjk.controller;

import cn.dev33.satoken.annotation.SaCheckPermission;
import jakarta.servlet.http.HttpServletResponse;
import jakarta.validation.constraints.NotNull;
import lombok.RequiredArgsConstructor;
import org.dromara.common.core.domain.R;
import org.dromara.common.core.validate.AddGroup;
import org.dromara.common.core.validate.EditGroup;
import org.dromara.common.excel.utils.ExcelUtil;
import org.dromara.common.idempotent.annotation.RepeatSubmit;
import org.dromara.common.log.annotation.Log;
import org.dromara.common.log.enums.BusinessType;
import org.dromara.common.mybatis.core.page.PageQuery;
import org.dromara.common.mybatis.core.page.TableDataInfo;
import org.dromara.common.web.core.BaseController;
import org.dromara.zjk.domain.bo.ZjkItemExtreationBo;
import org.dromara.zjk.domain.vo.ZjkItemExtreationVo;
import org.dromara.zjk.service.IZjkItemExtreationService;
import org.dromara.zjk.zwy.annotation.DecryptOperation;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * 抽取
 * 前端访问路由地址为:/jyk/itemExtreation
 *
 * @author ygh
 * @date 2024-10-29
 */
@Validated
@RequiredArgsConstructor
@RestController
@RequestMapping("/itemExtreation")
public class ZjkItemExtreationController extends BaseController {

    private final IZjkItemExtreationService zjkItemExtreationService;

    /**
     * 查询抽取列表
     */
    @SaCheckPermission("project:extract:list")
    @GetMapping("/list")
    public TableDataInfo<ZjkItemExtreationVo> list(ZjkItemExtreationBo bo, PageQuery pageQuery) {
        bo.setEffective(1);
        return zjkItemExtreationService.queryPageList(bo, pageQuery);
    }

    /**
     * 通过规则查询  抽取记录
     * @param id
     * @return
     */
    @DecryptOperation
    @GetMapping("extraction/{id}")
    public R<ZjkItemExtreationVo> getInfo1(@NotNull(message = "主键不能为空")
                                          @PathVariable Long id) {
        return R.ok(zjkItemExtreationService.queryExtractionById(id));
    }


    /**
     * 导出抽取列表
     */
    @Log(title = "抽取", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public void export(ZjkItemExtreationBo bo, HttpServletResponse response) {
        List<ZjkItemExtreationVo> list = zjkItemExtreationService.queryList(bo);
        ExcelUtil.exportExcel(list, "抽取", ZjkItemExtreationVo.class, response);
    }

    /**
     * 获取抽取详细信息
     *
     * @param id 主键
     */
    @GetMapping("/{id}")
    public R<ZjkItemExtreationVo> getInfo(@NotNull(message = "主键不能为空")
                                     @PathVariable Long id) {
        return R.ok(zjkItemExtreationService.queryById(id));
    }

    /**
     * 新增【请填写功能名称】
     */
    @SaCheckPermission("jyk:itemExtreation:add")
    @Log(title = "抽取", businessType = BusinessType.INSERT)
    @RepeatSubmit()
    @PostMapping()
    public R<Void> add(@Validated(AddGroup.class) @RequestBody ZjkItemExtreationBo bo) {
        return toAjax(zjkItemExtreationService.insertByBo(bo));
    }

    /**
     * 抽取确认
     */
    @Log(title = "抽取确认", businessType = BusinessType.UPDATE)
    @RepeatSubmit()
    @PostMapping("/edit")
    public R<Void> edit(@Validated(EditGroup.class) @RequestBody ZjkItemExtreationBo bo) {
        return toAjax(zjkItemExtreationService.updateByBo(bo));
    }

    /**
     * 获取抽取详情
     *
     * @param id 主键
     */
    @GetMapping("/queryOne/{id}")
    public R<ZjkItemExtreationVo> queryOne(@NotNull(message = "主键不能为空")
                                          @PathVariable Long id ) {
        return R.ok(zjkItemExtreationService.queryById(id));
    }


}
