package org.dromara.zjk.controller;

import java.util.List;

import cn.dev33.satoken.annotation.SaCheckRole;
import cn.dev33.satoken.annotation.SaMode;
import lombok.RequiredArgsConstructor;
import jakarta.servlet.http.HttpServletResponse;
import jakarta.validation.constraints.*;
import cn.dev33.satoken.annotation.SaCheckPermission;
import org.dromara.common.core.constant.TenantConstants;
import org.springframework.web.bind.annotation.*;
import org.springframework.validation.annotation.Validated;
import org.dromara.common.idempotent.annotation.RepeatSubmit;
import org.dromara.common.log.annotation.Log;
import org.dromara.common.web.core.BaseController;
import org.dromara.common.mybatis.core.page.PageQuery;
import org.dromara.common.core.domain.R;
import org.dromara.common.core.validate.AddGroup;
import org.dromara.common.core.validate.EditGroup;
import org.dromara.common.log.enums.BusinessType;
import org.dromara.common.excel.utils.ExcelUtil;
import org.dromara.zjk.domain.vo.ZjkLearningCreditRecordVo;
import org.dromara.zjk.domain.bo.ZjkLearningCreditRecordBo;
import org.dromara.zjk.service.IZjkLearningCreditRecordService;
import org.dromara.common.mybatis.core.page.TableDataInfo;

/**
 * 信用用户学习记录
 * 前端访问路由地址为:/jyk/learningCreditRecord
 *
 * @author ygh
 * @date 2024-11-26
 */
@Validated
@RequiredArgsConstructor
@RestController
@RequestMapping("/learningCreditRecord")
public class ZjkLearningCreditRecordController extends BaseController {

    private final IZjkLearningCreditRecordService zjkLearningCreditRecordService;

    /**
     * 查询信用用户学习记录列表
     */
    @GetMapping("/list")
    public TableDataInfo<ZjkLearningCreditRecordVo> list(ZjkLearningCreditRecordBo bo, PageQuery pageQuery) {
        return zjkLearningCreditRecordService.queryPageList(bo, pageQuery);
    }

    /**
     * 导出信用用户学习记录列表
     */
    @Log(title = "信用用户学习记录", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public void export(ZjkLearningCreditRecordBo bo, HttpServletResponse response) {
        List<ZjkLearningCreditRecordVo> list = zjkLearningCreditRecordService.queryList(bo);
        ExcelUtil.exportExcel(list, "信用用户学习记录", ZjkLearningCreditRecordVo.class, response);
    }

    /**
     * 获取信用用户学习记录详细信息
     *
     * @param userId 用户id
     */
    @GetMapping("/{expertId}")
    public R<List<ZjkLearningCreditRecordVo>> getInfo(@NotNull(message = "主键不能为空")
                                     @PathVariable Long userId) {
        return R.ok(zjkLearningCreditRecordService.queryById(userId));
    }

    /**
     * 新增信用用户学习记录
     */
    @Log(title = "信用用户学习记录", businessType = BusinessType.INSERT)
    @RepeatSubmit()
    @PostMapping("/add")
    public R<Void> add(@Validated(AddGroup.class) @RequestBody ZjkLearningCreditRecordBo bo) {
        return toAjax(zjkLearningCreditRecordService.insertByBo(bo));
    }

    /**
     * 修改信用用户学习记录
     */
    @Log(title = "信用用户学习记录", businessType = BusinessType.UPDATE)
    @RepeatSubmit()
    @PostMapping("/edit")
    public R<Void> edit(@Validated(EditGroup.class) @RequestBody ZjkLearningCreditRecordBo bo) {
        return toAjax(zjkLearningCreditRecordService.updateByBo(bo));
    }

    /**
     * 删除信用用户学习记录
     *
     * @param ids 主键串
     */
    @Log(title = "信用用户学习记录", businessType = BusinessType.DELETE)
    @GetMapping("/delete/{ids}")
    public R<Void> remove(@NotEmpty(message = "主键不能为空")
                          @PathVariable Long[] ids) {
        return toAjax(zjkLearningCreditRecordService.deleteWithValidByIds(List.of(ids), true));
    }
}
