package org.dromara.zjk.controller;

import cn.dev33.satoken.annotation.SaCheckPermission;
import cn.dev33.satoken.annotation.SaCheckRole;
import cn.dev33.satoken.annotation.SaMode;
import jakarta.validation.constraints.NotNull;
import lombok.RequiredArgsConstructor;
import org.dromara.common.core.constant.TenantConstants;
import org.dromara.common.core.domain.R;
import org.dromara.common.core.validate.AddGroup;
import org.dromara.common.idempotent.annotation.RepeatSubmit;
import org.dromara.common.log.annotation.Log;
import org.dromara.common.log.enums.BusinessType;
import org.dromara.common.mybatis.core.page.PageQuery;
import org.dromara.common.mybatis.core.page.TableDataInfo;
import org.dromara.common.web.core.BaseController;
import org.dromara.zjk.domain.bo.ZjkLeaveBo;
import org.dromara.zjk.domain.bo.ZjkLeaveRuleBo;
import org.dromara.zjk.domain.vo.*;
import org.dromara.zjk.service.IZjkLeaveRuleService;
import org.dromara.zjk.service.IZjkLeaveService;
import org.springframework.beans.propertyeditors.CustomDateEditor;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.*;

import java.text.SimpleDateFormat;
import java.util.Date;

/**
 * 专家库请假控制器
 *
 * @author : yanguohua
 * @version : 1.0
 * @date : 2024/10/30 16:57
 */
@RequiredArgsConstructor
@RestController
@RequestMapping("/leave")
public class ZjkLeaveController extends BaseController {
    private final IZjkLeaveService zjkLeaveService;

    private final IZjkLeaveRuleService zjkLeaveRuleService;
    @InitBinder  //解决前段传过来时间的字符串解析成时间报错问题
    public void initBinder(WebDataBinder webDataBinder){
        webDataBinder.registerCustomEditor(Date.class,new CustomDateEditor(new SimpleDateFormat("yyyy-MM-dd"),true));
    }

    @GetMapping("/pros")
    @SaCheckRole(value = {
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY
    }, mode = SaMode.OR)
    public TableDataInfo<ZjkEvaluateVO> getLeavePros() {
        return zjkLeaveService.getLeavePros();
    }

    /**
     * 待接受项目
     * @return
     */
    @GetMapping("/prosForAccept")
    @SaCheckRole(value = {
        TenantConstants.EXPERT_ROLE_KEY
    }, mode = SaMode.OR)
    public TableDataInfo<ZjkEvaluateVO> getProsForAccept() {
        return zjkLeaveService.getProsForAccept();
    }

    /**
     * 待参评项目
     * @return
     */
    @GetMapping("/prosForJoin")
    @SaCheckRole(value = {
        TenantConstants.EXPERT_ROLE_KEY
    }, mode = SaMode.OR)
    public TableDataInfo<ZjkEvaluateVO> getProsForJoin() {
        return zjkLeaveService.getProsForJoin();
    }
    /**
     * 项目请假
     *
     * @return
     */
    @PostMapping("/leaveForPro")
    @SaCheckRole(value = {
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY
    }, mode = SaMode.OR)
    public R<Void> leaveForPro(@RequestBody ZjkEvaluateVO bo) {
        return zjkLeaveService.leaveForPro(bo);
    }

    /**
     * 项目接受
     *
     * @return
     */
    @PostMapping("/setAccept")
    @SaCheckRole(TenantConstants.EXPERT_ROLE_KEY)
    public R<Void> setAccept(@RequestBody ZjkEvaluateVO bo) {
        return zjkLeaveService.setAccept(bo);
    }

    /**
     * 查询【请填写功能名称】列表
     */
    @GetMapping("/list")
    @SaCheckPermission("leave:log:list")
    @SaCheckRole(TenantConstants.EXPERT_ROLE_KEY)
    public TableDataInfo<ZjkLeaveVo> list(ZjkLeaveBo bo, PageQuery pageQuery) {
        return zjkLeaveService.queryPageList(bo, pageQuery);
    }

    /**
     * 查询【请填写功能名称】列表
     */
    @GetMapping("/adminList")
    @SaCheckPermission("leave:list:list")
    @SaCheckRole(TenantConstants.JG_ROLE_KEY)
    public TableDataInfo<ZjkLeaveVo> adminList(ZjkLeaveBo bo, PageQuery pageQuery) {
        return zjkLeaveService.queryAdminPageList(bo, pageQuery);
    }

    /**
     * 机构管理员-请假审批
     */
    @GetMapping("/splist")
    @SaCheckPermission("leave:approval:list")
    @SaCheckRole(TenantConstants.JG_ROLE_KEY)
    public TableDataInfo<ZjkLeaveVo> splist(ZjkLeaveBo bo, PageQuery pageQuery) {
        return zjkLeaveService.queryPageSpList(bo, pageQuery);
    }

    /**
     * 获取【请填写功能名称】详细信息
     *
     * @param gid 主键
     */
    @GetMapping("/detail/{gid}")
    @SaCheckPermission("leave:detail:page")
    public R<ZjkLeaveVo> getInfo(@NotNull(message = "主键不能为空")
                                 @PathVariable String gid) {
        return R.ok(zjkLeaveService.queryById(gid));
    }

    /**
     * 获取【请填写功能名称】详细信息
     */
    @GetMapping("/getRule")
    @SaCheckPermission("leave:rule:page")
    @SaCheckRole(TenantConstants.JG_ROLE_KEY)
    public R<ZjkLeaveRuleVo> getLeavelRule() {
        return R.ok(zjkLeaveRuleService.selectRuleByTannet());
    }

    /**
     * 修改【请填写功能名称】
     */
    @PutMapping("/setRule")
    @SaCheckRole(TenantConstants.JG_ROLE_KEY)
    public R<Void> edit(@RequestBody ZjkLeaveRuleBo bo) {
        return toAjax(zjkLeaveRuleService.updateByBo(bo));
    }

    /**
     * 新增【请填写功能名称】
     */
    @Log(title = "专家请假申请", businessType = BusinessType.INSERT)
    @SaCheckRole(TenantConstants.EXPERT_ROLE_KEY)
    @RepeatSubmit()
    @PostMapping()
    public R<Void> add(@Validated(AddGroup.class) @RequestBody ZjkLeaveBo bo) {
        return zjkLeaveService.insertByBo(bo);
    }

    /**
     * 修改【请填写功能名称】
     */
    @PutMapping()
    @SaCheckRole(TenantConstants.JG_ROLE_KEY)
    public R<Void> edit(@RequestBody ZjkLeaveBo bo) {
        return toAjax(zjkLeaveService.updateByBo(bo));
    }

    /**
     * 请假次数统计
     */
    @GetMapping("/stsNum")
    @SaCheckRole(value = {
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY
    }, mode = SaMode.OR)
    public R<StsNumVO> stsNum() {
        return zjkLeaveService.getStsNum();
    }
    /**
     * 请假图表统计
     */
    @GetMapping("/stsChart")
    @SaCheckRole(value = {
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY
    }, mode = SaMode.OR)
    public R<StsLeaveVO> stsChart(Integer rangeMonth) {
        return zjkLeaveService.getStsChart(rangeMonth);
    }
}
