package org.dromara.zjk.controller;

import java.util.List;

import cn.dev33.satoken.annotation.SaCheckRole;
import cn.dev33.satoken.annotation.SaMode;
import lombok.RequiredArgsConstructor;
import jakarta.servlet.http.HttpServletResponse;
import jakarta.validation.constraints.*;
import cn.dev33.satoken.annotation.SaCheckPermission;
import org.dromara.common.core.constant.TenantConstants;
import org.springframework.web.bind.annotation.*;
import org.springframework.validation.annotation.Validated;
import org.dromara.common.idempotent.annotation.RepeatSubmit;
import org.dromara.common.log.annotation.Log;
import org.dromara.common.web.core.BaseController;
import org.dromara.common.mybatis.core.page.PageQuery;
import org.dromara.common.core.domain.R;
import org.dromara.common.core.validate.AddGroup;
import org.dromara.common.core.validate.EditGroup;
import org.dromara.common.log.enums.BusinessType;
import org.dromara.common.excel.utils.ExcelUtil;
import org.dromara.zjk.domain.vo.ZjkProductReviewsnVo;
import org.dromara.zjk.domain.bo.ZjkProductReviewsnBo;
import org.dromara.zjk.service.IZjkProductReviewsnService;
import org.dromara.common.mybatis.core.page.TableDataInfo;

/**
 * 项目阶段文档管理
 * 前端访问路由地址为:/zjk/productReviewsn
 *
 * @author wt
 * @date 2024-11-22
 */
@Validated
@RequiredArgsConstructor
@RestController
@RequestMapping("/productReviewsn")
public class ZjkProductReviewsnController extends BaseController {

    private final IZjkProductReviewsnService zjkProductReviewsnService;

    /**
     * 查询项目阶段文档管理列表
     */
    @SaCheckRole(value = {
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    },mode = SaMode.OR)
    @GetMapping("/list")
    public TableDataInfo<ZjkProductReviewsnVo> list(ZjkProductReviewsnBo bo, PageQuery pageQuery) {
        return zjkProductReviewsnService.queryPageList(bo, pageQuery);
    }

    /**
     * 导出项目阶段文档管理列表
     */
    @SaCheckPermission("zjk:productReviewsn:export")
    @SaCheckRole(value = {
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "项目阶段文档管理", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public void export(ZjkProductReviewsnBo bo, HttpServletResponse response) {
        List<ZjkProductReviewsnVo> list = zjkProductReviewsnService.queryList(bo);
        ExcelUtil.exportExcel(list, "项目阶段文档管理", ZjkProductReviewsnVo.class, response);
    }

    /**
     * 获取项目阶段文档管理详细信息
     *
     * @param id 主键
     */
    @SaCheckPermission("zjk:productReviewsn:query")
    @SaCheckRole(value = {
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    },mode = SaMode.OR)
    @GetMapping("/{id}")
    public R<ZjkProductReviewsnVo> getInfo(@NotNull(message = "主键不能为空")
                                     @PathVariable Long id) {
        return R.ok(zjkProductReviewsnService.queryById(id));
    }

    /**
     * 新增项目阶段文档管理
     */
    @SaCheckPermission("zjk:productReviewsn:add")
    @SaCheckRole(value = {
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "项目阶段文档管理", businessType = BusinessType.INSERT)
    @RepeatSubmit()
    @PostMapping("/add")
    public R<Void> add(@Validated(AddGroup.class) @RequestBody ZjkProductReviewsnBo bo) {
        return toAjax(zjkProductReviewsnService.insertByBo(bo));
    }

    /**
     * 修改项目阶段文档管理
     */
    @SaCheckPermission("zjk:productReviewsn:edit")
    @SaCheckRole(value = {
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "项目阶段文档管理", businessType = BusinessType.UPDATE)
    @RepeatSubmit()
    @PostMapping("/edit")
    public R<Void> edit(@Validated(EditGroup.class) @RequestBody ZjkProductReviewsnBo bo) {
        return toAjax(zjkProductReviewsnService.updateByBo(bo));
    }

    /**
     * 删除项目阶段文档管理
     *
     * @param ids 主键串
     */
    @SaCheckPermission("zjk:productReviewsn:remove")
    @SaCheckRole(value = {
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "项目阶段文档管理", businessType = BusinessType.DELETE)
    @GetMapping("/delete/{ids}")
    public R<Void> remove(@NotEmpty(message = "主键不能为空")
                          @PathVariable Long[] ids) {
        return toAjax(zjkProductReviewsnService.deleteWithValidByIds(List.of(ids), true));
    }
}
