package org.dromara.zjk.controller;

import java.util.List;

import cn.dev33.satoken.annotation.SaCheckRole;
import cn.dev33.satoken.annotation.SaMode;
import lombok.RequiredArgsConstructor;
import jakarta.servlet.http.HttpServletResponse;
import jakarta.validation.constraints.*;
import cn.dev33.satoken.annotation.SaCheckPermission;
import org.dromara.common.core.constant.TenantConstants;
import org.dromara.zjk.domain.vo.AuditVo;
import org.springframework.web.bind.annotation.*;
import org.springframework.validation.annotation.Validated;
import org.dromara.common.idempotent.annotation.RepeatSubmit;
import org.dromara.common.log.annotation.Log;
import org.dromara.common.web.core.BaseController;
import org.dromara.common.mybatis.core.page.PageQuery;
import org.dromara.common.core.domain.R;
import org.dromara.common.core.validate.AddGroup;
import org.dromara.common.core.validate.EditGroup;
import org.dromara.common.log.enums.BusinessType;
import org.dromara.common.excel.utils.ExcelUtil;
import org.dromara.zjk.domain.vo.ZjkProductSummarizeLogVo;
import org.dromara.zjk.domain.bo.ZjkProductSummarizeLogBo;
import org.dromara.zjk.service.IZjkProductSummarizeLogService;
import org.dromara.common.mybatis.core.page.TableDataInfo;

/**
 * 总结日志表
 * 前端访问路由地址为:/zjk/productSummarizeLog
 *
 * @author ygh
 * @date 2024-11-04
 */
@Validated
@RequiredArgsConstructor
@RestController
@RequestMapping("/productSummarizeLog")
public class ZjkProductSummarizeLogController extends BaseController {

    private final IZjkProductSummarizeLogService zjkProductSummarizeLogService;

    /**
     * 查询总结日志表列表
     */
    @SaCheckRole(value =
        {
            TenantConstants.XM_ROLE_KEY,
            TenantConstants.TENANT_ADMIN_ROLE_KEY,
        }
        , mode = SaMode.OR)
    @GetMapping("/list")
    public TableDataInfo<ZjkProductSummarizeLogVo> list(ZjkProductSummarizeLogBo bo, PageQuery pageQuery) {
        return zjkProductSummarizeLogService.queryPageList(bo, pageQuery);
    }

    /**
     * 导出总结日志表列表
     */
    @Log(title = "总结日志表", businessType = BusinessType.EXPORT)
    @SaCheckRole(value =
        {
            TenantConstants.XM_ROLE_KEY,
            TenantConstants.TENANT_ADMIN_ROLE_KEY,
        }
        , mode = SaMode.OR)
    @PostMapping("/export")
    public void export(ZjkProductSummarizeLogBo bo, HttpServletResponse response) {
        List<ZjkProductSummarizeLogVo> list = zjkProductSummarizeLogService.queryList(bo);
        ExcelUtil.exportExcel(list, "总结日志表", ZjkProductSummarizeLogVo.class, response);
    }

    /**
     * 获取总结日志表详细信息
     *
     * @param id 主键
     */
    @SaCheckRole(value = {
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
    }, mode = SaMode.OR)
    @GetMapping("/{id}")
    public R<ZjkProductSummarizeLogVo> getInfo(@NotNull(message = "主键不能为空")
                                               @PathVariable Long id) {
        return R.ok(zjkProductSummarizeLogService.queryById(id));
    }

    /**
     * 新增总结日志表
     */
    @Log(title = "总结日志表", businessType = BusinessType.INSERT)
    @RepeatSubmit()
    @SaCheckPermission("zjk:productSummarizeLog:add")
    @PostMapping()
    public R<Void> add(@Validated(AddGroup.class) @RequestBody ZjkProductSummarizeLogBo bo) {
        return toAjax(zjkProductSummarizeLogService.insertByBo(bo));
    }

    /**
     * 修改总结日志表
     */
    @Log(title = "总结日志表", businessType = BusinessType.UPDATE)
    @RepeatSubmit()
    @SaCheckPermission("zjk:productSummarizeLog:update")
    @PutMapping()
    public R<Void> edit(@Validated(EditGroup.class) @RequestBody ZjkProductSummarizeLogBo bo) {
        return toAjax(zjkProductSummarizeLogService.updateByBo(bo));
    }

    /**
     * 删除总结日志表
     *
     * @param ids 主键串
     */
    @SaCheckRole(value =
        {
            TenantConstants.XM_ROLE_KEY,
            TenantConstants.TENANT_ADMIN_ROLE_KEY,
            TenantConstants.JG_ROLE_KEY,
        }
        , mode = SaMode.OR)
    @Log(title = "总结日志表", businessType = BusinessType.DELETE)
    @DeleteMapping("/{ids}")
    public R<Void> remove(@NotEmpty(message = "主键不能为空")
                          @PathVariable Long[] ids) {
        return toAjax(zjkProductSummarizeLogService.deleteWithValidByIds(List.of(ids), true));
    }

    @SaCheckRole(value =
        {
            TenantConstants.XM_ROLE_KEY,
            TenantConstants.TENANT_ADMIN_ROLE_KEY,
            TenantConstants.JG_ROLE_KEY,
        }
        , mode = SaMode.OR)
    @Log(title = "审核", businessType = BusinessType.UPDATE)
    @PostMapping("/audit")
    public R<Void> edit(@RequestBody AuditVo bo) {
        return toAjax(zjkProductSummarizeLogService.audit(bo));
    }
}
